<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;

class Post extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'user_id',
        'association_id',
        'title',
        'description',
        'link',
        'like_count',
        'comment_count',
        'status',
        'send_notifications'
    ];

    /**
     * @return BelongsTo
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    /**
     * @return BelongsTo
     */
    public function association(): BelongsTo
    {
        return $this->belongsTo(User::class, 'association_id');
    }

    /**
     * @return HasMany
     */
    public function medias(): HasMany
    {
        return $this->hasMany(Media::class, 'post_id');
    }

    /**
     * @return HasMany
     */
    public function comments(): HasMany
    {
        return $this->hasMany(PostComment::class, 'post_id');
    }

    public function saveComment(User $user, $comment)
    {
        return $this->comments()->create([
            'user_id' => $user->id,
            'comment' => $comment,
        ]);
    }

    public function updateComment(PostComment $postComment, $newComment)
    {
        $postComment->update([
            'comment' => $newComment,
        ]);

        return $postComment;
    }

    /**
     * @return HasMany
     */
    public function postLikes(): HasMany
    {
        return $this->hasMany(PostLike::class, 'post_id');
    }

    /**
     * @return BelongsToMany
     */
    public function likedByUser(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'post_likes', 'post_id', 'user_id')
            ->withTimestamps();
    }

    public function isLiked(User $user): bool
    {
        return $this->postLikes()->where('user_id', $user->id)->exists();
    }

    // Unlike the post
    public function removeLike(User $user)
    {
        $like = $this->postLikes()->where('user_id', $user->id)->first();
        if ($like) {
            $like->delete();
        }
    }

    // Like the post
    public function like(User $user)
    {
        if (!$this->postLikes()->where('user_id', $user->id)->exists()) {
            $this->postLikes()->create([
                'user_id' =>  $user->id
            ]);
        }
    }

    public function likedByUsers(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'post_likes', 'post_id', 'user_id')
            ->withTimestamps();
    }

    public function isLikedByUser(User $user)
    {
        return $this->likedByUsers()->where('user_id', $user->id)->exists() ? 1 : 0;
    }


    public function bookmarkedByUsers(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'post_bookmarks', 'post_id', 'user_id')
            ->withTimestamps();
    }

    public function isBookmarkedByUser(User $user)
    {
        return $this->bookmarkedByUsers()->where('user_id', $user->id)->exists() ? 1 : 0;
    }

    // bookmark post
    public function bookmarkPost(User $user)
    {
        if (!$this->bookmarkedByUsers()->where('user_id', $user->id)->exists()) {
            $this->bookmarkedByUsers()->attach($user);
        }
    }

    public function removeBookmark(User $user)
    {
        $this->bookmarkedByUsers()->detach($user);
    }

    public function isBookmarked(User $user): bool
    {
        return $this->bookmarkedByUsers()->where('user_id', $user->id)->exists();
    }
}
