<?php

namespace App\Models;

use App\Enums\LanguageEnum;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;

class MessageTemplate extends Model
{
    use HasFactory;

    protected $fillable = [
        'template_id',
        'name',
        'language',
        'category',
        'allow_category_change',
        'schedule_type',
        'schedule_date',
        'schedule_time',
        'status',
        'association_id',
    ];

    public function headerComponent(): HasOne
    {
        return $this->hasOne(MessageTemplateComponent::class, 'message_template_id')
            ->where('type', 'HEADER');
    }

    public function bodyComponent(): HasOne
    {
        return $this->hasOne(MessageTemplateComponent::class, 'message_template_id')
            ->where('type', 'BODY');
    }

    public function footerComponent(): HasOne
    {
        return $this->hasOne(MessageTemplateComponent::class, 'message_template_id')
            ->where('type', 'FOOTER');
    }

    public function buttonComponents(): HasMany
    {
        return $this->hasMany(MessageTemplateComponent::class, 'message_template_id')
            ->whereNotIn('type', ['HEADER', 'BODY', 'FOOTER']);
    }

    public function components(): HasMany
    {
        return $this->hasMany(MessageTemplateComponent::class, 'message_template_id');
    }
    public function getLanguageContentAttribute()
    {
        $languages = LanguageEnum::LANGUAGES;
        if (isset($languages[$this->language])) {
            return $languages[$this->language];
        } else {
            return '-';
        }
    }

    const FAILED = "FAILED";
    const APPROVED = 'APPROVED';
    const PENDING = 'PENDING';
    const REJECTED = 'REJECTED';

    const CATEGORIES = [
        'AUTHENTICATION',
        'MARKETING',
        'UTILITY'
    ];

    const HEADER_TYPES = [
        'None',
        'TEXT',
        'MEDIA',
    ];

    const MEDIA_TYPES = [
        'Image',
        'Video',
        'Document',
    ];

    const BUTTON_TYPES = [
        'None',
        'PHONE_NUMBER',
        'URL',
        'QUICK_REPLY',
    ];

    public function getFacebookPayload()
    {
        $components = [];
        $headerComponent = $this->headerComponent()->first();
        if ($headerComponent) {
            $components[] = $headerComponent->getFacebookPayload();
        }

        $bodyComponent = $this->bodyComponent()->first();
        $components[] = $bodyComponent->getFacebookPayload();

        $footerComponent = $this->footerComponent()->first();
        if ($footerComponent) {
            $components[] = $footerComponent->getFacebookPayload();
        }

        $buttonComponents = [];
        foreach ($this->buttonComponents()->where("type", "!=", "QUICK_REPLY")->get() as $buttonComponent) {
            $buttonComponents[] = $buttonComponent->getFacebookPayload();
        }

        foreach ($this->buttonComponents()->where("type", "QUICK_REPLY")->get() as $buttonComponent) {
            $buttonComponents[] = $buttonComponent->getFacebookPayload();
        }

        if (count($buttonComponents) > 0) {
            $components[] = [
                "type" => "BUTTONS",
                "buttons" => $buttonComponents
            ];
        }

        return [
            "name" => $this->name,
            "category" => $this->category,
            "allow_category_change" => $this->allow_category_change,
            "language" => $this->language,
            "components" => $components,
        ];
    }

    public function createMessage($host, User $user)
    {
        $message = '<div class="preview-wrap"><div class="card preview-card"><div class="card-body">';

        //header message
        $headerComponents = $this->headerComponent;
        if (isset($headerComponents)) {
            if ($headerComponents->format == 'IMAGE') {
                $path = $headerComponents->media->path ? getFilePath($headerComponents->media->path)  : '';
                $message .= '<div class="card-header"><img class="img-fluid" src="' . $path  . '"></div>';
            }

            if ($headerComponents->format == 'VIDEO') {
                $path = $headerComponents->media->path ? getFilePath($headerComponents->media->path)  : '';
                $message .= '<div class="card-header"><video controls><source src="' . $path . '" type="video/mp4"></video></div>';
            }

            if ($headerComponents->format == 'DOCUMENT') {
                $path = $headerComponents->media->path ? getFilePath($headerComponents->media->path)  : '';
                $message .= '<div class="card-header"><iframe src="'. $path .'" style="width:100% !important"></iframe></div>';
            }

            if ($headerComponents->format == 'TEXT') {
                $headerMessage = $headerComponents->text;
                $dynamicVariables = $headerComponents->dynamic_variables ?? [];
                foreach ($dynamicVariables as $key => $dynamicVariable) {
                    $headerMessage = str_replace('[ ' . $key . ' ]', $user->$key, $headerMessage);
                }
                $message .= '<h2 class="title-1">' . $headerMessage . '</h2>';
            }
        }

        //body message
        $bodyComponents = $this->bodyComponent;
        if (isset($bodyComponents)) {
            $bodyMessage = $bodyComponents->text;
            $dynamicVariables = $bodyComponents->dynamic_variables ?? [];
            if (isset($dynamicVariables)) {
                foreach ($dynamicVariables as $key => $dynamicVariable) {
                    $bodyMessage = str_replace('[ ' . $key . ' ]', $user->$key, $bodyMessage);
                }
            }
            $message .= $bodyMessage;
        }
        $message .= '</div>';

        //footer message
        if (isset($this->footerComponents)) {
            $message .= '<div class="card-footer">';
            $message .= '<p class="footnote">' . $this->footerComponents->text . '</p></div>';
        }
        $message .= '</div>';

        //button message
        $buttonComponents = $this->buttonComponents;
        if (isset($buttonComponents) && count($buttonComponents) > 0) {
            $message .= '<div class="card-button">';
            foreach ($buttonComponents as $button) {
                if ($button->type == 'URL') {
                    $message .= '<a href="' . $button->format . '" target="_blank" class="btn btn-white-shadow"><svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-geo-alt-fill" viewBox="0 0 16 16"><path d="M8 16s6-5.686 6-10A6 6 0 0 0 2 6c0 4.314 6 10 6 10zm0-7a3 3 0 1 1 0-6 3 3 0 0 1 0 6z" /></svg>' . $button->text . '</a>';
                } elseif ($button->type == 'PHONE_NUMBER') {
                    $message .= '<a href="tel:' . $button->format . '" target="_blank" class="btn btn-white-shadow"><svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-telephone-fill" viewBox="0 0 16 16"><path fill-rule="evenodd" d="M1.885.511a1.745 1.745 0 0 1 2.61.163L6.29 2.98c.329.423.445.974.315 1.494l-.547 2.19a.678.678 0 0 0 .178.643l2.457 2.457a.678.678 0 0 0 .644.178l2.189-.547a1.745 1.745 0 0 1 1.494.315l2.306 1.794c.829.645.905 1.87.163 2.611l-1.034 1.034c-.74.74-1.846 1.065-2.877.702a18.634 18.634 0 0 1-7.01-4.42 18.634 18.634 0 0 1-4.42-7.009c-.362-1.03-.037-2.137.703-2.877L1.885.511z" /></svg>' . $button->text . '</a>';
                } else {
                    $message .= '<button class="btn btn-white-shadow">' . $button->text . '</button>';
                }
            }
            $message .= '</div>';
        }

        $message .= '</div>';
        return $message;
    }
}
