<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Builder;

class EventVenue extends Model
{
    use HasFactory;

    protected $fillable = [
        'event_day_id',
        'name',
        'start_date_time',
        'end_date_time',
        'latitude',
        'longitude',
        'contact',
        'website_link',
        'index',
        'extra',
    ];
    protected $casts = [
        'extra' => 'array',
        'start_date_time' => 'datetime:H:i',
        'end_date_time' => 'datetime:H:i',
    ];

    protected $hidden = [
        "created_at",
        "updated_at",
    ];

    // Global scope to order by index ascending
    protected static function booted()
    {
        static::addGlobalScope('orderByIndex', function (Builder $builder) {
            $builder->orderBy('index');
        });
    }

    // Accessor to format start_date_time and end_date_time as 'H:i'
    public function getStartDateTimeAttribute($value)
    {
        return $value ? \Carbon\Carbon::parse($value)->format('H:i') : null;
    }
    public function getEndDateTimeAttribute($value)
    {
        return $value ? \Carbon\Carbon::parse($value)->format('H:i') : null;
    }

    /**
     * @return BelongsTo
     */
    public function eventDay(): BelongsTo
    {
        return $this->belongsTo(EventDay::class, 'event_day_id');
    }

    public function scrollerImages()
    {
        return $this->morphMany(ScrollerImage::class, 'imageable');
    }

}
