<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Builder;

class EventSession extends Model
{
    use HasFactory;

    //TYPE
    const TYPE_AUTO = 1;
    const TYPE_MANUAL = 2;


    protected $fillable = [
        'sub_sessions_min',
        'sub_sessions',
        'start_time',
        'parent_id',
        'end_time',
        'event_id',
        'event_hall_id',
        'event_day_id',
        'title',
        'topic',
        'type',
        'text_color',
        'bg_color',
        'created_at',
        'updated_at',
        'notes',
        'status',
        'ask_questions',
        'moderator',
    ];

    protected $hidden = [
        'created_at',
        'updated_at',
    ];

    // Global scope to order by start_time ascending
    protected static function booted()
    {
        static::addGlobalScope('orderByStartTime', function (Builder $builder) {
            $builder->orderBy('start_time');
        });
    }

    /**
     * @return BelongsTo
     */
    public function event(): BelongsTo
    {
        return $this->BelongsTo(Event::class, 'event_id');
    }

    /**
     * @return BelongsTo
     */
    public function eventHall(): BelongsTo
    {
        return $this->BelongsTo(EventHall::class, 'event_hall_id');
    }

    /**
     * @return BelongsTo
     */
    public function eventDay(): BelongsTo
    {
        return $this->BelongsTo(EventDay::class, 'event_day_id');
    }

    public function eventSessionUsers(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'event_session_users')
            ->withPivot('event_designation_id')
            ->withTimestamps();
    }

    public function eventSessionUsersDetails(): BelongsToMany
    {
        return $this->belongsToMany(EventSession::class, 'event_session_users')->withPivot(['event_designation_id', 'title'])
            ->withTimestamps();
    }

    public function eventSessionFaculties(): BelongsToMany
    {
        return $this->belongsToMany(EventFaculty::class, 'event_session_users')
            ->withTimestamps();
    }

    public function eventSessionSpeakers(): BelongsToMany
    {
        return $this->belongsToMany(EventSpeaker::class, 'event_session_users', 'event_session_id', 'event_speaker_id')
            ->withTimestamps();
    }

    /**
     * @return HasMany
     */
    public function subSessions(): HasMany
    {
        return $this->hasMany($this, 'parent_id');
    }

    /**
     * @return BelongsTo
     */
    public function parentSession(): BelongsTo
    {
        return $this->BelongsTo($this, 'parent_id');
    }

    /**
     * @return HasMany
     */
    public function questions(): HasMany
    {
        return $this->hasMany(EventSessionQuestion::class, 'event_session_id');
    }

    // Accessor to format start_time and end_time as 'H:i'
    public function getStartTimeAttribute($value)
    {
        return $value ? \Carbon\Carbon::parse($value)->format('H:i') : null;
    }
    public function getEndTimeAttribute($value)
    {
        return $value ? \Carbon\Carbon::parse($value)->format('H:i') : null;
    }

    // event session bookmark related function
    public function bookmarkedByUsers(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'event_session_bookmarks', 'event_session_id', 'user_id')
            ->withTimestamps();
    }

    public function isBookmarkedByUser(User $user)
    {
        return $this->bookmarkedByUsers()->where('user_id', $user->id)->exists() ? 1 : 0;
    }

    public function bookmarkEventSession(User $user, $eventId)
    {
        if (!$this->bookmarkedByUsers()->where('user_id', $user->id)->exists()) {
            $this->bookmarkedByUsers()->attach($user->id, ['event_id' => $eventId]);
        }
    }

    public function removeBookmark(User $user)
    {
        $this->bookmarkedByUsers()->detach($user);
    }

    public function isBookmarked(User $user): bool
    {
        return $this->bookmarkedByUsers()->where('user_id', $user->id)->exists();
    }
}
