<?php

namespace App\Jobs\Meta;

use App\Helpers\MetaClient;
use App\Models\Message;
use App\Models\MessageCampaignUser;
use App\Models\MessageTemplate;
use App\Models\User;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class SendCampaignMessage implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    use MetaLog;

    protected $host;

    protected $campaignUserId;

    /**
     * Create a new job instance.
     *
     * @param $host
     * @param $campaignUserId
     */
    public function __construct($host, $campaignUserId)
    {
        $this->host = $host;

        $this->campaignUserId = $campaignUserId;
    }

    /**
     * @return MessageCampaignUser
     */
    public function getCampaignUser()
    {
        return MessageCampaignUser::find($this->campaignUserId);
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle()
    {

        $campaignUser = $this->getCampaignUser();
        $user = $campaignUser->user;
        $template = $campaignUser->campaign->messageTemplate;

        $payload = $this->getTemplateComponents($this->host, $template, $user);

        $log = $this->createLog("send_message_template", $payload);

        $api = new MetaClient();
        $response = $api->sendTemplateMessage($user->full_mobile_number, $template->name, $template->language, $payload);

        $json = $response->json();
        $this->saveResponse($log, $json);

        if ($response->ok()) {
            $this->logInfo($json);
            $campaignUser->contact = $user->full_mobile_number;
            $campaignUser->status = MessageCampaignUser::STATUS_SENT;
            $campaignUser->meta_message_id = $json["messages"][0]["id"] ?? null;
            $campaignUser->save();

            $message = $template->createMessage($this->host, $user);

            Message::create([
                "receiver_id" => $user->id,
                "message_template_id" => $template->id,
                "message" => $message,
                "message_campaign_id" => $campaignUser->message_campaign_id ?? null,
                "message_id" => $json["messages"][0]["id"] ?? null,
            ]);
            $campaignUser->campaign->handleSuccess($user->id);
        } else {
            $this->logError($json);
            $campaignUser->contact = $user->full_mobile_number;
            $campaignUser->status = MessageCampaignUser::STATUS_FAILED;
            $campaignUser->save();
            $campaignUser->campaign->handleFailed($user->id);
        }
    }

    protected function getTemplateComponents($host, MessageTemplate $template, User $user)
    {
        $components = [];

        // header parameters
        $header = $template->headerComponent;
        if ($header) {
            $headerComponent = [
                "type" => "header",
                "parameters" => []
            ];
            if ($header->format == "IMAGE") {
                $headerComponent['parameters'][] = [
                    "type" => "image",
                    "image" => [
                        "link" => $host . "/storage/" . $header->media->path
                    ]
                ];
                $components[] = $headerComponent;
            } else if ($header->format == "DOCUMENT") {
                $headerComponent['parameters'][] = [
                    "type" => "document",
                    "document" => [
                        "link" => $host . "/storage/" . $header->media->path,
                        "filename" => basename($header->media->path)
                    ]
                ];
                $components[] = $headerComponent;
            } else if ($header->format == "VIDEO") {
                $headerComponent['parameters'][] = [
                    "type" => "video",
                    "video" => [
                        "link" => $host . "/storage/" . $header->media->path,
                    ]
                ];
                $components[] = $headerComponent;
            }

            if (!empty($header->example_key) && !empty($header->dynamic_variables)) {
                if ($header->format == 'TEXT') {
                    $variables = $header->dynamic_variables ?? [];
                    foreach ($variables as $key => $variable) {
                        $metaProperty = "meta_{$key}";
                        $headerComponent['parameters'][] = $user->$metaProperty;
                    }
                }
                $components[] = $headerComponent;
            }
        }

        // body parameters
        $body = $template->bodyComponent;
        $bodyComponent = ["type" => "body", "parameters" => []];
        if ($body) {
            $variables = $body->dynamic_variables ?? [];
            foreach ($variables as $key => $variable) {
                $metaProperty = "meta_{$key}";
                $bodyComponent["parameters"][] = $user->$metaProperty;
            }
        }
        $components[] = $bodyComponent;
        return $components;
    }
}
