<?php

namespace App\Imports;

use App\Enums\RoleEnum;
use App\Models\AssociationUser;
use App\Models\MemberDetail;
use App\Models\User;
use App\Rules\DisposableEmailRule;
use Illuminate\Support\Collection;
use Maatwebsite\Excel\Concerns\ToCollection;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Hash;

class AssociationMemberImport implements ToCollection
{
    private $associationId;
    public function __construct($associationId)
    {
        $this->associationId = $associationId;
    }
    /**
     * @param Collection $collection
     */
    public function collection(Collection $rows)
    {
        foreach ($rows as $key => $row) {
            if ($key === 0 || $this->shouldSkipRow($row)) {
                continue;
            }

            if (!filter_var($row[1], FILTER_VALIDATE_EMAIL)) {
                Log::info('Invalid Email: ' . $row[1]);
                continue;
            }

            // if (User::where('email', $row[1])->exists()) {
            //     Log::info('Email Already Exists: ' . $row[1]);
            //     $this->linkUserToAssociation($user, $row);
            //     continue;
            // }

            $user = $this->createOrUpdateUser($row);
            $this->linkUserToAssociation($user, $row);
            $this->createOrUpdateMemberDetail($user, $row);
        }
    }

    protected function shouldSkipRow($row): bool
    {
        $disposableEmailRule = new DisposableEmailRule();

        return empty($row[0]) ||
            empty($row[1]) ||
            $disposableEmailRule->isDisposable($row[1]);
    }

    protected function createOrUpdateUser($row)
    {
        return User::updateOrCreate(
            ['email' => $row[1]],
            [
                'association_id' => $this->associationId,
                'name' => $row[0] ?? '',
                'contact' => $this->sanitizePhoneNumber($row[2] ?? ''),
                'password' => Hash::make('123456'),
                'role' => RoleEnum::MEMBER,
            ]
        );
    }

    protected function linkUserToAssociation(User $user, $row): void
    {
        AssociationUser::updateOrCreate(
            [
                'user_id' => $user->id,
                'association_id' => $this->associationId,
            ],
            [
                'user_member_id' => $row[5] ?? '',
            ]
        );
    }

    protected function createOrUpdateMemberDetail(User $user, $row): void
    {
        $userDetail = [
            'member_id' => $user->id,
            'association_id' => $this->associationId,
            'about' => $row[3] ?? '',
            'cv' => $row[4] ?? '',
        ];

        // Handle registration number uniqueness
        if (!empty($row[6])) {
            $existing = MemberDetail::where('registration_number', $row[6])
                ->where('member_id', '!=', $user->id)
                ->first();

            if ($existing) {
                Log::info('Duplicate Registration Number: ' . $row[6], [
                    'member_id' => $existing->id ?? '',
                    'user_id' => $existing->user->id ?? '',
                    'user_name' => $existing->user->name ?? '',
                    'user_email' => $existing->user->email ?? '',
                ]);
            } else {
                $userDetail['registration_number'] = $row[6];
            }
        }

        $memberDetail = $user->memberDetail;

        if ($memberDetail) {
            $memberDetail->update($userDetail);
        } else {
            $user->memberDetail()->create($userDetail);
        }
    }

    protected function sanitizePhoneNumber(string $raw): int
    {
        return (int) substr(preg_replace('/\D/', '', str_replace(' ', '', $raw)), -10);
    }
}
