<?php

namespace App\Http\Controllers\Event;

use App\Enums\StatusEnum;
use App\Http\Controllers\Controller;
use App\Http\Traits\MediaUpload;
use App\Models\EventDay;
use App\Models\EventVenue;
use App\Models\ScrollerImage;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;

class VenueController extends Controller
{
    use MediaUpload;

    public function index(Request $request, EventDay $eventDay)
    {
        $eventVenues = $eventDay->eventVenues();
        if ($request->filter) {
            $eventVenues->where(function ($q) use ($request) {
                $q->orWhere('name', 'like', "%{$request->filter}%");
            });
        }

        $eventVenues = $eventVenues->paginate(25);
        return view('event.venue.index', compact('eventVenues', 'eventDay', 'request'));
    }

    public function create(EventDay $eventDay)
    {
        return view('event.venue.add', compact('eventDay'));
    }

    public function store(Request $request, EventDay $eventDay)
    {
        $request->validate([
            'name' => ['required', 'string'],
            'country_code' => ['required'],
            'contact' => ['nullable', 'digits:10'],
            'video_link' => ['nullable', 'url'],
            'website_link' => ['nullable', 'url'],
            'index' => ['nullable', 'integer'],
            'address' => ['nullable', 'string'],
            'address_link' => ['nullable', 'url'],
            'latitude' => ['nullable'],
            'longitude' => ['nullable'],
            'start_time' => ['nullable'],
            'end_time' => ['nullable'],
        ]);
        $data = $request->except(['_token', '_method']);

        if ($request->start_time) {
            $data['start_date_time'] = date("H:i", strtotime($request->start_time));
        }
        if ($request->end_time) {
            $data['end_date_time'] = date("H:i", strtotime($request->end_time));
        }
        $data['country_code'] = "+" . str_replace('+', '', $data['country_code']);
        $data["index"] = $request->index ?? EventVenue::max('index') + 1;
        $data['extra'] = [
            'description' => $request->get('description', ''),
            'address' => $request->get('address', ''),
            'address_link' => $request->get('address_link', ''),
            'video_link' => $request->get('video_link', ''),
        ];
        $eventDay->eventVenues()->create($data);
        return redirect()->route('event.venues.index', $eventDay->id)->with('success', 'Speaker Created');
    }

    public function edit(EventVenue $eventVenue)
    {
        return view('event.venue.edit', compact('eventVenue'));
    }

    public function update(EventVenue $eventVenue, Request $request)
    {
        $request->validate([
            'name' => ['required', 'string'],
            'country_code' => ['required'],
            'contact' => ['nullable', 'digits:10'],
            'video_link' => ['nullable', 'url'],
            'website_link' => ['nullable', 'url'],
            'index' => ['nullable', 'integer'],
            'address' => ['nullable', 'string'],
            'address_link' => ['nullable', 'url'],
            'latitude' => ['nullable'],
            'longitude' => ['nullable'],
            'start_time' => ['nullable'],
            'end_time' => ['nullable'],
        ]);

        $data = $request->except(['_token', '_method']);

        if ($request->start_time) {
            $data['start_date_time'] = date("H:i", strtotime($request->start_time));
        }
        if ($request->end_time) {
            $data['end_date_time'] = date("H:i", strtotime($request->end_time));
        }

        $data['country_code'] = "+" . str_replace('+', '', $data['country_code']);
        $data["index"] = $request->index ?? EventVenue::max('index') + 1;
        $data['extra'] = [
            'description' => $request->get('description', ''),
            'address' => $request->get('address', ''),
            'address_link' => $request->get('address_link', ''),
            'video_link' => $request->get('video_link', ''),
        ];
        $eventVenue->update($data);
        return redirect()->route('event.venues.index', $eventVenue->eventDay->id ?? '')->with('success', 'Speaker Updated');
    }

    public function statusChange(EventVenue $eventVenue)
    {
        $newStatus = ($eventVenue->status == StatusEnum::ACTIVE) ? StatusEnum::INACTIVE : StatusEnum::ACTIVE;
        $eventVenue->status = $newStatus;
        $eventVenue->save();
        return response()->json(["status" => 200, "message" => "Speaker Status Changed"]);
    }

    public function updateIndex(Request $request, EventVenue $eventVenue)
    {
        $request->validate([
            'index' => 'required',
        ]);
        try {
            $eventVenue->index = $request->index;
            $eventVenue->update();

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => 'success',
                    'message' => 'Index change successfully',
                ]);
            }
            return redirect()->route('event.venue.index', $eventVenue->eventDay->id ?? '');
        } catch (\Exception $e) {
            return redirect()->back()->with(['error', $e->getMessage()]);
        }
    }

    public function destroy(EventVenue $eventVenue)
    {
        $eventVenue->delete();
        return response()->json(["status" => 200, "message" => "Speaker Deleted"]);
    }

    public function imagesIndex(EventVenue $eventVenue)
    {
        $title = $eventVenue->name;
        $addRoute = route('event.venues.images.add', $eventVenue->id);
        $scrollImages = $eventVenue->scrollerImages()->paginate(25);
        return view('event.scroll-images.index', compact('scrollImages', 'eventVenue', 'title', 'addRoute'));
    }

    public function imageAdd(EventVenue $eventVenue)
    {
        $title = $eventVenue->name;
        $storeRoute = route('event.venues.images.store', $eventVenue->id);
        return view('event.scroll-images.add', compact('eventVenue', 'title', 'storeRoute'));
    }

    public function imagestore(Request $request, EventVenue $eventVenue)
    {
        $request->validate([
            'image' => ['required', 'image', 'max:2048'],
        ]);
        $scrollerImage = new ScrollerImage([
            'index' => $request->index ?? 0,
            'media_id' => $this->upload($request->image)->id,
        ]);
        $eventVenue->scrollerImages()->save($scrollerImage);
        return redirect()->route('event.venues.images.index', $eventVenue->id)->with('success', 'Image Created');
    }
}
