<?php

namespace App\Http\Controllers\Event;

use App\Enums\StatusEnum;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Event;
use App\Models\EventHall;
use Illuminate\Support\Facades\Session;
use Illuminate\Validation\Rule;

class EventHallController extends Controller
{
    public function index(Request $request)
    {
        $eventHalls = EventHall::where('event_id', Session::get('event_id'));
        $eventHalls = $eventHalls->paginate(25);
        return view('event.hall.index', compact('request', 'eventHalls'));
    }

    public function store(Request $request, Event $event)
    {
        $data = $request->validate([
            'name' => ['required', 'string'],
            'status' => ['required'],
            'text_color' => ['required', 'regex:/^#+([a-fA-F0-9]{6}|[a-fA-F0-9]{3})$/'],
            'bg_color' => ['required', 'regex:/^#+([a-fA-F0-9]{6}|[a-fA-F0-9]{3})$/'],
        ]);

        if ($event->eventHalls()->where('name', $data['name'])->exists()) {
            return redirect()->back()->with('error', 'Hall Already Exists')->withInput();
        }

        $event->eventHalls()->create($data);
        return redirect()->route('event.halls.index')->with('success', 'Hall Created');
    }

    public function update(Request $request, EventHall $eventHall)
    {
        $data = $request->validate([
            'name' => ['required', 'string', Rule::unique('users')->ignore($eventHall->id),],
            'status' => ['required'],
            'text_color' => ['required', 'regex:/^#+([a-fA-F0-9]{6}|[a-fA-F0-9]{3})$/'],
            'bg_color' => ['required', 'regex:/^#+([a-fA-F0-9]{6}|[a-fA-F0-9]{3})$/'],
        ]);

        $eventId = Session::get('event_id');

        if (
            EventHall::where('name', $data['name'])
                ->where('event_id', $eventId)->whereNot('id', $eventHall->id)->exists()
        ) {
            return redirect()->back()->with('error', 'Hall Already Exists')->withInput();
        }

        $eventHall->update($data);
        return redirect()->route('event.halls.index')->with('success', 'Hall Updated');
    }

    public function statusChange(EventHall $eventHall)
    {
        $eventHall->status = ($eventHall->status == StatusEnum::ACTIVE) ? StatusEnum::INACTIVE : StatusEnum::ACTIVE;
        $eventHall->save();
        return response()->json(["status" => 200, "message" => "Hall Status Changed"]);
    }

    public function destroy(EventHall $eventHall)
    {
        $eventHall->delete();
        return response()->json(["status" => 200, "message" => "Hall Deleted"]);
    }
}
