<?php

namespace App\Http\Controllers\Event;

use App\Http\Controllers\Controller;
use App\Http\Traits\MediaUpload;
use App\Imports\EventFacultyImport;
use App\Models\Event;
use App\Models\EventFaculty;
use App\Rules\DisposableEmailRule;
use Illuminate\Support\Facades\Session;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Facades\Excel;

class EventFacultyController extends Controller
{
    use MediaUpload;

    public function index(Request $request)
    {
        $eventId = Session::get('event_id');
        $faculties = EventFaculty::where('event_id', $eventId);

        if ($request->filter) {
            $faculties->where(function ($q) use ($request) {
                $requestContact = preg_replace("/[^0-9]/", "", $request->filter);
                $q->where('name', 'like', "%{$request->filter}%")
                    ->orWhereLike('contact', $requestContact);
            });
        }
        if ($request->email) {
            $faculties->where(function ($q) use ($request) {
                $q->orWhere('email', $request->email);
            });
        }

        $faculties = $faculties->paginate(25);
        return view('event.faculty.index', compact('faculties', 'request'));
    }

    public function create()
    {
        return view('event.faculty.add');
    }

    public function store(Request $request, Event $event)
    {
        $data = $request->validate([
            'title' => ['required'],
            'name' => ['required', 'string'],
            'email' => ['required', 'email', 'unique:event_faculties,email', new DisposableEmailRule()],
            'country_code' => ['required'],
            'contact' => ['required', 'digits:10'],
            'designation' => ['nullable'],
            'personal_info' => ['nullable'],
            'institute' => ['nullable'],
            'text_1' => ['nullable'],
            'text_2' => ['nullable'],
            'bio' => ['nullable'],
            'image' => ['nullable', 'image', 'max:3072'],
        ]);

        if (isset($request->image)) {
            $data['media_id'] = $this->upload($request->image)->id;
        }
        $data['country_code'] = "+" . str_replace('+', '', $data['country_code']);


        $event->eventFaculties()->create($data);
        return redirect()->route('event.faculty.index')->with('success', 'Event Faculty Created');
    }

    public function edit(EventFaculty $faculty)
    {
        return view('event.faculty.edit', compact('faculty'));
    }

    public function update(EventFaculty $faculty, Request $request)
    {
        $data = $request->validate([
            'title' => ['required'],
            'name' => ['required', 'string'],
            'email' => ['required', 'email', 'unique:event_faculties,email,' . $faculty->id, new DisposableEmailRule()],
            'country_code' => ['required'],
            'contact' => ['required', 'digits:10'],
            'designation' => ['nullable'],
            'personal_info' => ['nullable'],
            'institute' => ['nullable'],
            'text_1' => ['nullable'],
            'text_2' => ['nullable'],
            'bio' => ['nullable'],
            'image' => ['nullable', 'image', 'max:3072'],
        ]);

        if (isset($request->image)) {
            $data['media_id'] = $this->upload($request->image)->id;
        }
        $data['country_code'] = "+" . str_replace('+', '', $data['country_code']);

        $faculty->update($data);
        return redirect()->route('event.faculty.index')->with('success', 'Event Faculty Updated');
    }

    public function destroy(EventFaculty $faculty)
    {
        $faculty->delete();
        return response()->json(["status" => 200, "message" => "Faculty Deleted"]);
    }

    public function import(Request $request, $eventId)
    {
        $request->validate([
            'file' => ['required', 'file'],

        ]);
        Excel::import(new EventFacultyImport($eventId), $request->file('file')->store('app'));

        return redirect()->back()->with('success', 'File imported successfully!');
    }

}
