<?php

namespace App\Http\Controllers\Event;

use App\Enums\AdminLevel;
use App\Http\Controllers\Controller;
use App\Models\Admin;
use App\Models\Country;
use App\Models\EventManager;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Session;
use Illuminate\Validation\Rule;

class EventCoordinatorController extends Controller
{
    public function index(Request $request)
    {
        $event = getEventFromId(Session::get('event_id'));
        $eventCoordinators = $event->eventManagers();
        $eventCoordinators = $eventCoordinators->filter();
        $eventCoordinators = $eventCoordinators->paginate(25);
        return view('event.event-coordinator.index', compact('eventCoordinators', 'request'));
    }

    public function create()
    {
        $countryCodes = Country::get();
        return view('event.event-coordinator.add', compact('countryCodes'));
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'name' => ['required', 'string'],
            'country_code' => ['required'],
            'contact' => ['required', 'integer', 'digits:10'],
            'permission_level' => ['required', "in:" . implode(',', \App\Enums\EventManagerEnum::getPrivileges())],
            'email' => 'required|unique:admins,email',
            'password' => ['required'],
        ]);

        $event = getEventFromId(Session::get('event_id'));

        $data['password'] = Hash::make($request->password);
        $data['role'] = AdminLevel::EVENT_COORDINATOR;
        $data["association_id"] = $event->association_id;
        $data['pin'] = $request->password;
        $data['country_code'] = "+" . $data['country_code'];
        $eventCoordinator = Admin::create($data);

        EventManager::create(
            [
                'event_id' => Session::get('event_id'),
                'association_id' => $event->association_id,
                'admin_id' => $eventCoordinator->id,
            ]
        );

        return redirect()->route('event.coordinators.index')->with('success', 'Event Coordinator Created Successfully');
    }

    public function edit(Admin $eventCoordinator)
    {
        $countryCodes = Country::get();
        return view('event.event-coordinator.edit', compact('eventCoordinator', 'countryCodes'));
    }

    public function update(Admin $eventCoordinator, Request $request)
    {
        $data = $request->validate([
            'name' => ['required', 'string'],
            'country_code' => ['required'],
            'contact' => ['required'],
            'email' => [
                'required',
                Rule::unique('admins')->ignore($eventCoordinator->id),
            ],
            'permission_level' => ['required', "in:" . implode(',', \App\Enums\EventManagerEnum::getPrivileges())],
            'password' => ['nullable'],
        ]);

        $data['pin'] = $data['password'];
        $data['password'] = Hash::make($data['password']);
        $data['country_code'] = "+" . $data['country_code'];

        $eventCoordinator->update($data);
        return redirect()->route('event.coordinators.index')->with('success', 'Event Coordinator Updated Successfully');
    }

    public function destroy(Admin $eventCoordinator)
    {
        $eventCoordinator->delete();
        return response()->json(["status" => 200, "message" => "Event Coordinator Deleted Successfully"]);
    }
}
