<?php

namespace App\Http\Controllers\Event;

use App\Enums\StatusEnum;
use App\Http\Controllers\Controller;
use App\Http\Traits\MediaUpload;
use App\Models\Event;
use App\Models\EventCommittee;
use App\Models\EventCommitteeMember;
use App\Models\EventMemberRole;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;

class EventCommitteeController extends Controller
{
    use MediaUpload;

    public function index(Request $request)
    {
        $eventId = Session::get('event_id');
        $committees = EventCommittee::where('event_id', $eventId);
        $committees = $committees->paginate(25);

        return view('event.committees.index', compact('committees', 'request'));
    }

    public function create()
    {
        return view('event.committees.add');
    }


    public function store(Request $request, Event $event)
    {
        $request->validate([
            'name' => ['required'],
        ]);

        $data = $request->except(['_token', '_method']);
        $data["event_id"] = $event->id;
        $data["index"] = $request->index ?? EventCommittee::max('index') + 1;
        EventCommittee::create($data);

        return redirect()->route('event.committees.index')->with('success', 'Committee Created');
    }

    public function edit(EventCommittee $committee)
    {
        return view('event.committees.edit', compact('committee'));
    }

    public function update(EventCommittee $committee, Request $request)
    {
        $request->validate([
            'name' => ['required'],
        ]);

        $data = $request->except(['_token', '_method']);
        $data["index"] = $request->index ?? 1;
        $committee->update($data);
        return redirect()->route('event.committees.index')->with('success', 'Committee Updated');
    }

    public function updateCommitteeIndex(Request $request, EventCommittee $committee)
    {
        $request->validate([
            'index' => 'required',
        ]);
        try {
            $committee->index = $request->index;
            $committee->update();

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => 'success',
                    'message' => 'Index change successfully',
                ]);
            }
            return redirect()->route('event.committees.index', $committee->id);
        } catch (\Exception $e) {
            return redirect()->back()->with(['error', $e->getMessage()]);
        }
    }

    public function view(EventCommittee $committee, Request $request)
    {
        $eventId = Session::get('event_id');
        $event = Event::find($eventId);

        $members = User::whereHas('memberDetail', function ($q) use ($event) {
            $q->where('association_id', $event->association_id);
        })
            ->where('status', StatusEnum::ACTIVE)
            ->get();

        $committeeMembers = EventCommitteeMember::where("committee_id", $committee->id)->whereHas('member');

        if ($request->name) {
            $committeeMembers->whereHas('member', function ($query) use ($request) {
                $query->where("name", 'like', '%' . $request->name . '%');
            });
        }
        if ($request->contact) {
            $committeeMembers->whereHas('member', function ($query) use ($request) {
                $query->where("contact", 'like', '%' . $request->contact . '%');
            });
        }

        $committeeMembers = $committeeMembers->get();

        if ($request->export == "export") {

            $fileName = $committee->name . ".csv";
            $headers = [
                "Content-type"        => "text/csv",
                "Content-Disposition" => "attachment; filename=$fileName",
                "Pragma"              => "no-cache",
                "Cache-Control"       => "must-revalidate, post-check=0, pre-check=0",
                "Expires"             => "0"
            ];

            $columns = ['Sr No.', 'Name', 'Contact', 'Role', 'About', 'Date'];

            // Callback function for CSV generation
            $callback = function () use ($committeeMembers, $columns) {
                $file = fopen('php://output', 'w');
                fputcsv($file, $columns);
                $i = 1;  // Start from 1 to begin numbering from 2nd row
                foreach ($committeeMembers as $committeeMember) {
                    fputcsv($file, [
                        $i,
                        $committeeMember->member->name ?? '-',
                        $committeeMember->member->contact ?? '-',
                        $committeeMember->role->name ?? '-',
                        $committeeMember->member->memberDetail->about ?? '-',
                        $committeeMember->member->created_at ? $committeeMember->member->created_at->timezone('Asia/Kolkata')->format('d-m-Y | h:i A') : '',
                    ]);
                    $i++;
                }
                fclose($file);
            };

            return response()->stream($callback, 200, $headers);
        }


        $memberRoles = EventMemberRole::where('event_id', $eventId)->whereStatus(StatusEnum::ACTIVE)->get();

        return view('event.committees.view', compact('committee', 'members', 'committeeMembers', 'memberRoles', 'request'));
    }

    public function createMember(EventCommittee $committee, Request $request)
    {
        $eventId = Session::get('event_id');
        $event = Event::find($eventId);
        $users = [];

        if (count($request->all()) > 0) {
            $users = User::where('status', StatusEnum::ACTIVE)->whereHas('memberDetail', function ($q) use ($event) {
                $q->where('association_id', $event->association_id);
            })->whereDoesntHave('eventCommitteeMember', function ($q) use ($committee) {
                $q->where('committee_id', $committee->id);
            });

            if (!empty($request->name) && $request->name != 'all') {
                $users->where('name', 'like', "%" . $request->name . "%");
            }

            if (!empty($request->mobile) && $request->mobile != 'all') {
                $users->where('contact', 'like', "%" . $request->mobile . "%");
            }

            if (!empty($request->status) && $request->status != 'all') {
                if ($request->status == "inactive") {
                    $statusData = StatusEnum::INACTIVE;
                }
                if ($request->status == "active") {
                    $statusData = StatusEnum::ACTIVE;
                }
                $users->where('status', $statusData);
            }

            $users = $users->get();
        }

        return view('event.committees.add-members', compact('committee', 'users', 'request'));
    }


    public function addMember(EventCommittee $committee, Request $request)
    {
        $eventId = Session::get('event_id');
        $request->validate([
            'member_ids' => ['required'],
        ]);

        $data = $request->except(['_token', '_method']);

        foreach ($data['member_ids'] as $memberId) {
            $user = EventCommitteeMember::where('committee_id', $committee->id)->where('member_id', $memberId)->first();
            if (!$user) {
                EventCommitteeMember::create([
                    'event_id' => $eventId,
                    'committee_id' => $committee->id,
                    'member_id' => $memberId,
                    'index' => EventCommitteeMember::max('index') + 1,
                ]);
            }
        }

        return redirect()->route('event.committees.view', $committee->id)->with('success', 'Committee Member Added');
    }

    public function destroyMember(EventCommitteeMember $committeeMember)
    {
        $committeeMember->delete();
        return response()->json(["status" => 200, "message" => "Member Deleted"]);
    }

    public function statusChange(EventCommittee $committee)
    {
        if ($committee->status == StatusEnum::ACTIVE) {
            $committee->status = StatusEnum::INACTIVE;
        } else {
            $committee->status = StatusEnum::ACTIVE;
        }
        $committee->save();
        return response()->json(["status" => 200, "message" => "Status Changed"]);
    }

    public function destroy(EventCommittee $committee)
    {
        $committee->delete();
        return response()->json(["status" => 200, "message" => "Committee Deleted"]);
    }

    public function updateIndex(Request $request, EventCommitteeMember $committeeMember)
    {
        $request->validate([
            'index' => 'required',
        ]);
        try {
            $committeeMember->index = $request->index;
            $committeeMember->update();

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => 'success',
                    'message' => 'Index change successfully',
                ]);
            }
            return redirect()->route('event.committees.view', $committeeMember->id);
        } catch (\Exception $e) {
            return redirect()->back()->with(['error', $e->getMessage()]);
        }
    }

    public function updateRole(Request $request, EventCommitteeMember $committeeMember)
    {
        $request->validate([
            'role_id' => 'required',
        ]);
        try {
            $committeeMember->role_id = $request->role_id;
            $committeeMember->update();

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => 'success',
                    'message' => 'Role change successfully',
                ]);
            }
            return redirect()->route('event.committees.view', $committeeMember->id);
        } catch (\Exception $e) {
            return redirect()->back()->with(['error', $e->getMessage()]);
        }
    }
}
