<?php

namespace App\Http\Controllers\Event;

use App\Enums\ButtonEnum;
use App\Enums\StatusEnum;
use App\Http\Controllers\Controller;
use App\Http\Traits\MediaUpload;
use App\Models\Event;
use App\Models\EventAppHomeButton;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;

class EventAppHomeButtonController extends Controller
{
    use MediaUpload;

    public function index()
    {
        $homeButtons = EventAppHomeButton::where('event_id', Session::get('event_id'));
        $homeButtons = $homeButtons->paginate(25);
        return view('event.app.home-button.index', compact('homeButtons'));
    }

    public function create(Request $request)
    {
        return view('event.app.home-button.create');
    }

    public function store(Request $request, Event $event)
    {
        $data = $request->validate([
            'name' => ['required', 'string'],
            'link' => ['required'],
            'text_color' => ['nullable', 'regex:/^#+([a-fA-F0-9]{6}|[a-fA-F0-9]{3})$/'],
            'bg_color' => ['nullable', 'regex:/^#+([a-fA-F0-9]{6}|[a-fA-F0-9]{3})$/'],
            'image' => ['nullable', 'image', 'max:3072'],
            'index' => ['nullable'],
            'status' => ['required'],
        ]);

        $data = $request->except(['_token', '_method']);

        if (($request->file('image'))) {
            $data['media_id'] = $this->upload($request->image)->id;
        }
        $key = Str::slug($request->name, '_');
        $data['key'] = Str::slug($request->name, '_');
        $data['type'] = ButtonEnum::DYNAMIC;
        $data["index"] = $request->index ?? EventAppHomeButton::max('index') + 1;

        $event->eventAppHomeButtons()->create($data);
        return redirect()->route('event.app.home.buttons.index')->with('success', 'Button Created');
    }

    public function edit(EventAppHomeButton $homeButton)
    {
        return view('event.app.home-button.edit', compact('homeButton'));
    }

    public function update(EventAppHomeButton $homeButton, Request $request)
    {
        $data = $request->validate([
            'name' => [
                Rule::requiredIf(function () use ($homeButton) {
                    return $homeButton->type === ButtonEnum::DYNAMIC;
                }),
                'string',
            ],
            'link' => ['nullable'],
            'text_color' => ['nullable', 'regex:/^#+([a-fA-F0-9]{6}|[a-fA-F0-9]{3})$/'],
            'bg_color' => ['nullable', 'regex:/^#+([a-fA-F0-9]{6}|[a-fA-F0-9]{3})$/'],
            'image' => ['nullable', 'image', 'max:3072'],
            'index' => ['nullable'],
            'status' => ['required'],
        ]);

        $data = $request->except(['_token', '_method']);

        if (($request->file('image'))) {
            $data['media_id'] = $this->upload($request->image)->id;
        }
        if ($request->get('name')) {
            $data['key'] = Str::slug($request->get('name'), '_');
        }
        $data["index"] = $request->index ?? EventAppHomeButton::max('index') + 1;

        $homeButton->update($data);
        return redirect()->route('event.app.home.buttons.index')->with('success', 'Button Updated');
    }

    public function statusChange(EventAppHomeButton $homeButton)
    {
        $newStatus = ($homeButton->status == StatusEnum::ACTIVE) ? StatusEnum::INACTIVE : StatusEnum::ACTIVE;
        $homeButton->status = $newStatus;
        $homeButton->save();
        return response()->json(["status" => 200, "message" => "Status Changed"]);
    }

    public function updateIndex(Request $request, EventAppHomeButton $homeButton)
    {
        $request->validate([
            'index' => 'required',
        ]);
        try {
            $homeButton->index = $request->index;
            $homeButton->update();

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => 'success',
                    'message' => 'Index change successfully',
                ]);
            }
            return redirect()->route('event.app.home.buttons.index');
        } catch (\Exception $e) {
            return redirect()->back()->with(['error', $e->getMessage()]);
        }
    }

    public function destroy(EventAppHomeButton $homeButton)
    {
        if ($homeButton->type == ButtonEnum::STATIC) {
            return response()->json(["status" => 400, "message" => "Unauthorized"]);
        }

        $homeButton->delete();
        return response()->json(["status" => 200, "message" => "Button Deleted"]);
    }

    public function addData()
    {
        return;
        $sessionID = Session::get('event_id');
        $data = [
            [
                "event_id" => $sessionID,
                "name" => "About Us",
                "key" => "about_us",
                "text_color" => "#ffffff",
                "bg_color" => "#000000"
            ],
            [
                "event_id" => $sessionID,
                "name" => "About City",
                "key" => "about_city",
                "text_color" => "#ffffff",
                "bg_color" => "#000000"
            ],
            [
                "event_id" => $sessionID,
                "name" => "Venue Details",
                "key" => "venue_details",
                "text_color" => "#ffffff",
                "bg_color" => "#000000"
            ],
            [
                "event_id" => $sessionID,
                "name" => "Committee",
                "key" => "committee",
                "text_color" => "#ffffff",
                "bg_color" => "#000000"
            ],
            [
                "event_id" => $sessionID,
                "name" => "Speakers",
                "key" => "speakers",
                "text_color" => "#ffffff",
                "bg_color" => "#000000"
            ],
            [
                "event_id" => $sessionID,
                "name" => "Scientific Programme",
                "key" => "scientific_programme",
                "text_color" => "#ffffff",
                "bg_color" => "#000000"
            ],
            [
                "event_id" => $sessionID,
                "name" => "Registration",
                "key" => "registration",
                "text_color" => "#ffffff",
                "bg_color" => "#000000"
            ],
            [
                "event_id" => $sessionID,
                "name" => "Exhibitors",
                "key" => "exhibitors",
                "text_color" => "#ffffff",
                "bg_color" => "#000000"
            ],
            [
                "event_id" => $sessionID,
                "name" => "Important Links",
                "key" => "important_links",
                "text_color" => "#ffffff",
                "bg_color" => "#000000"
            ],
            [
                "event_id" => $sessionID,
                "name" => "Document",
                "key" => "document",
                "text_color" => "#ffffff",
                "bg_color" => "#000000"
            ],
            [
                "event_id" => $sessionID,
                "name" => "Activity",
                "key" => "activity",
                "text_color" => "#ffffff",
                "bg_color" => "#000000"
            ],
            [
                "event_id" => $sessionID,
                "name" => "Contact Us",
                "key" => "contact_us",
                "text_color" => "#ffffff",
                "bg_color" => "#000000"
            ],
            [
                "event_id" => $sessionID,
                "name" => "Notification",
                "key" => "notification",
                "text_color" => "#ffffff",
                "bg_color" => "#000000"
            ],
        ];

        foreach ($data as $item) {
            EventAppHomeButton::updateOrCreate(
                [
                    'event_id' => $item['event_id'],
                    'name' => $item['name']
                ],
                $item,
            );
        }
        return 'Data added successfully';
    }
}
