<?php

namespace App\Http\Controllers\Association;

use App\Enums\StatusEnum;
use App\Http\Controllers\Controller;
use App\Http\Traits\MediaUpload;
use App\Models\Slider;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class SliderController extends Controller
{
    use MediaUpload;

    public function index(Request $request)
    {
        $sliders = Slider::where('association_id', Auth::user()->id)->orderBy('index');

        $sliders = $sliders->paginate(25);

        return view('association.sliders.index', compact('sliders', 'request'));
    }

    public function create()
    {
        return view('association.sliders.add');
    }

    public function store(Request $request)
    {
        $request->validate([
            'slider_image' => ['required', 'image', 'max:3072'],
        ]);

        $data = $request->except(['_token', '_method']);

        if (isset($request->slider_image)) {
            $data['media_id'] = $this->upload($request->slider_image)->id;
        }
        $data["association_id"] = Auth::user()->id;
        $data["index"] = $request->index ?? Slider::max('index') + 1;

        Slider::create($data);

        return redirect()->route('sliders.index')->with('success', 'Slider Created');
    }

    public function edit(Slider $slider)
    {
        return view('association.sliders.edit', compact('slider'));
    }

    public function update(Slider $slider, Request $request)
    {
        $request->validate([
            'slider_image' => ['nullable', 'image', 'max:3072'],
        ]);

        $data = $request->except(['_token', '_method']);
        if (isset($request->slider_image)) {
            $this->updateMedia($slider->media_id, $request->slider_image, $slider->image->path);
        }
        $data["index"] = $request->index ?? Slider::max('index') + 1;
        $slider->update($data);
        return redirect()->route('sliders.index')->with('success', 'Slider Updated');
    }

    public function statusChange(Slider $slider)
    {
        if ($slider->status == StatusEnum::ACTIVE) {
            $slider->status = StatusEnum::INACTIVE;
        } else {
            $slider->status = StatusEnum::ACTIVE;
        }
        $slider->save();
        return response()->json(["status" => 200, "message" => "Slider Status Changed"]);
    }

    public function destroy(Slider $slider)
    {
        $slider->delete();
        return response()->json(["status" => 200, "message" => "Slider Deleted"]);
    }
}
