<?php

namespace App\Http\Controllers\Association;

use App\Enums\PollOptionAnswerTypeEnum;
use App\Enums\PollOptionTypeEnum;
use App\Http\Controllers\Controller;
use App\Http\Traits\MediaUpload;
use App\Models\Poll;
use App\Models\PollOption;
use Illuminate\Http\Request;

class PollOptionController extends Controller
{
    use MediaUpload;

    public function index(Poll $poll)
    {
        $pollOptions = $poll->pollOptions()->paginate(20);
        return view('association.polls-option.index', compact('pollOptions', 'poll'));
    }

    public function create(Poll $poll)
    {
        return view('association.polls-option.create', compact('poll'));
    }

    public function store(Request $request, Poll $poll)
    {
        $data = $request->validate([
            'type' => "required|integer|in:" . PollOptionTypeEnum::TEXT . "," . PollOptionTypeEnum::IMAGE,
            'option' => 'required_if:type,' . PollOptionTypeEnum::TEXT,
            'image' => 'required_if:type,' . PollOptionTypeEnum::IMAGE . '|image|max:3048',
        ]);

        if (isset($request->image)) {
            $data['option'] = $this->upload($request->image)->id;
        }
        $data['association_id'] = $this->getCurrentUserId();
        
        $poll->pollOptions()->create($data);

        return redirect()->route('polls.options.index', $poll->id)->with('success', 'Poll option created successfully.');
    }

    public function edit(Request $request, PollOption $pollOption)
    {
        $poll = $pollOption->poll;
        return view('association.polls-option.edit', compact('poll', 'pollOption'));
    }

    public function update(Request $request, PollOption $pollOption)
    {
        $request->validate([
            'type' => "required|integer|in:" . PollOptionTypeEnum::TEXT . "," . PollOptionTypeEnum::IMAGE,
        ]);
        $data = $request->only(['type', 'option', 'image']);

        if (!$request->option && !$request->image) {
            $data['option'] = $pollOption->option;
        }

        if (isset($request->image)) {
            if (isset($pollOption->image->path)) {
                $this->remove($pollOption->image->path);
            }
            $data['option'] = $this->upload($request->image)->id;
        }

        $poll = $pollOption->poll;
        $pollOption->update($data);

        return redirect()->route('polls.options.index', $poll->id)->with('success', 'Poll option created successfully.');
    }

    public function typeChange(PollOption $pollOption)
    {
        $newType = ($pollOption->answer_type == PollOptionAnswerTypeEnum::RIGHT) ? PollOptionAnswerTypeEnum::WRONG : PollOptionAnswerTypeEnum::RIGHT;
        if ($newType == PollOptionAnswerTypeEnum::RIGHT) {
            PollOption::where('poll_id', $pollOption->poll_id)
                ->where('id', '!=', $pollOption->id)
                ->update(['answer_type' => PollOptionAnswerTypeEnum::WRONG]);
        }
        $pollOption->answer_type = $newType;
        $pollOption->save();
        return response()->json(["status" => 200, "message" => "Poll Option Type Changed"]);
    }

    public function destroy(PollOption $pollOption)
    {
        $pollOption->delete();
        return response()->json(["status" => 200, "message" => "Poll Option Deleted"]);
    }
}
