<?php

namespace App\Http\Controllers\Association;

use App\Enums\MembershipFormFieldEnum;
use App\Http\Controllers\Controller;
use App\Models\MembershipFormField;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\Rule;

class MembershipFormController extends Controller
{
    public function index(Request $request)
    {
        $membershipFormFields = MembershipFormField::where('association_id', Auth::user()->id)->orderBy('index')->get();
        return view('association.membership-form.index', compact('membershipFormFields'));
    }

    public function storeField(Request $request)
    {
        $request->validate([
            'label' => ['required', 'string'],
            'name' => ['required', 'string'],
            'index' => ['nullable', 'integer'],
            'type' => ['required', Rule::in(['text', 'date', 'textarea', 'radio', 'checkbox', 'dropdown', 'image'])],
            'field_accept_value' => ['nullable', Rule::in([MembershipFormFieldEnum::FIELD_ACCEPT_ALPHA_NUMBERIC, MembershipFormFieldEnum::FIELD_ACCEPT_ALPHABETS, MembershipFormFieldEnum::FIELD_ACCEPT_NUMBERIC])],
        ]);

        $inputField = [];

        $inputField["label"] = $request->get('label', null);
        $inputField["name"] = $request->get('name', null);
        $inputField["default_value"] = $request->get('default_value', null);
        $inputField["helper_text"] = $request->get('helper_text', null);
        $inputField["index"] = $request->index ?? MembershipFormField::max('index') + 1;
        $inputField["type"] = $request->get('type', null);
        $inputField["required"] = $request->get('required', "0");
        $inputField["field_accept_value"] = $request->get('field_accept_value', null);

        $nonOptionTypes = ['text', 'textarea', 'image', 'date'];
        $inputField["options"] = in_array($request->type, $nonOptionTypes) ? null : $this->getOptionsDetail($request->options);

        if ($request->type == 'image') {
            $inputField["field_accept_value"] = MembershipFormFieldEnum::FIELD_ACCEPT_IMAGE;
        }
        $inputField["association_id"] = Auth::user()->id;

        MembershipFormField::create($inputField);

        return redirect()->back()->with('success', 'DC Input Field Added');
    }

    public function editField(MembershipFormField $membershipFormField)
    {
        $eventField = [];
        $eventField['label'] = $membershipFormField->label;
        $eventField['index'] = $membershipFormField->index;
        $eventField['required'] = $membershipFormField->required;
        $eventField['helper_text'] = $membershipFormField->helper_text;
        $eventField['type'] = $membershipFormField->type;
        $eventField['options'] = $membershipFormField->options;
        $eventField['name'] = $membershipFormField->name;
        $eventField['field_accept_value'] = $membershipFormField->field_accept_value;
        $eventField['default_value'] = $membershipFormField->default_value;

        return response()->json($eventField);
    }

    public function getOptionsDetail(array $data, array $response = [])
    {
        $data = cleanseArray($data);
        if (count($data) > 0) {
            foreach ($data['value'] as $key => $details) {
                $response[$key]['value'] = $details;
                $response[$key]['title'] = $data['title'][$key];
            }
        }
        return $response;
    }

    public function updateFields(MembershipFormField $membershipFormField, Request $request)
    {
        $request->validate([
            'label' => ['required', 'string'],
            'index' => ['nullable', 'integer'],
            'type' => ['required', Rule::in(['text', 'date', 'textarea', 'radio', 'checkbox', 'dropdown', 'image'])],
            'field_accept_value' => ['nullable', Rule::in([MembershipFormFieldEnum::FIELD_ACCEPT_ALPHA_NUMBERIC, MembershipFormFieldEnum::FIELD_ACCEPT_ALPHABETS, MembershipFormFieldEnum::FIELD_ACCEPT_NUMBERIC])],
        ]);
        $inputField = [];
        $inputField["required"] = $request->get('required', 0);
        $inputField["label"] = $request->get('label', null);
        $inputField["helper_text"] = $request->get('helper_text', null);
        $inputField["index"] = $request->index ?? $membershipFormField->index;
        $inputField["field_accept_value"] = $request->get('field_accept_value', null);
        $inputField["default_value"] = $request->get('default_value', null);

        $nonOptionTypes = ['text', 'textarea', 'image', 'date'];
        $inputField["options"] = in_array($request->type, $nonOptionTypes) ? null : $this->getOptionsDetail($request->options);

        if ($request->type == 'image') {
            $inputField["field_accept_value"] = MembershipFormFieldEnum::FIELD_ACCEPT_IMAGE;
        }

        MembershipFormField::find($membershipFormField->id)->update($inputField);
        if ($request->ajax()) {
            return response()->json("success");
        } else {
            return redirect()->back()->with('success', 'DC Input Fields Updated');
        }
    }

    public function deleteField(MembershipFormField $membershipFormField)
    {
        $membershipFormField->delete();
        return response()->json("success");
    }
}
