<?php

namespace App\Http\Controllers\Association;

use App\Enums\RoleEnum;
use App\Enums\StatusEnum;
use App\Http\Controllers\Controller;
use App\Http\Traits\MediaUpload;
use App\Imports\AssociationMemberImport;
use App\Models\AssociationUser;
use App\Models\Country;
use App\Models\MemberDetail;
use App\Models\MemberRole;
use App\Models\User;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\Rule;
use Maatwebsite\Excel\Facades\Excel;

class MemberController extends Controller
{
    use MediaUpload;

    public function index(Request $request)
    {
        $association = Auth::user();
        $members = User::latest()->whereHas('memberDetail', function ($q) use ($request) {
            if ($request->role_id && $request->role_id != 'all') {
                $q->where('role_id', $request->role_id);
            }
        })
            ->whereHas('associationUsers', function ($q) use ($association) {
                $q->where('association_id', $association->id);
            })
            ->orderBy('id', 'desc');

        if ($request->filter) {
            $members->where(function ($query) use ($request) {
                $query->where('name', 'like', '%' . $request->filter . '%')
                    ->orWhere('email', 'like', '%' . $request->filter . '%');
            });
        }
        if ($request->contact) {
            $members->where("contact", 'like', '%' . $request->contact . '%');
        }
        if ($request->registration_number) {
            $members->whereHas('memberDetail', function ($q) use ($request) {
                $q->where('registration_number', 'like', '%' . $request->registration_number . '%');
            });
        }

        if ($request->export == "export") {
            $members = $members->get();

            $fileName = "members.csv";
            $headers = [
                "Content-type" => "text/csv",
                "Content-Disposition" => "attachment; filename=$fileName",
                "Pragma" => "no-cache",
                "Cache-Control" => "must-revalidate, post-check=0, pre-check=0",
                "Expires" => "0"
            ];

            $columns = ['Sr No.', 'Name', 'Email', 'Contact', 'Role', 'About', 'Date'];

            // Callback function for CSV generation
            $callback = function () use ($members, $columns) {
                $file = fopen('php://output', 'w');
                fputcsv($file, $columns);
                $i = 1;  // Start from 1 to begin numbering from 2nd row
                foreach ($members as $member) {
                    fputcsv($file, [
                        $i,
                        $member->name ?? '-',
                        $member->email ?? '-',
                        $member->contact ?? '-',
                        $member->memberDetail->role->name ?? '-',
                        $member->memberDetail->about ?? '-',
                        $member->created_at->format('d-m-Y | h:i A'),
                    ]);
                    $i++;
                }
                fclose($file);
            };

            return response()->stream($callback, 200, $headers);
        }

        $members = $members->paginate(25);

        $memberRoles = MemberRole::where('association_id', $association->id)->get();

        return view('association.members.index', compact('members', 'request', 'memberRoles', 'association'));
    }

    public function create()
    {
        $memberRoles = MemberRole::where('association_id', Auth::user()->id)->where('status', StatusEnum::ACTIVE)->get();
        $countryCodes = Country::get();
        return view('association.members.add', compact('countryCodes', 'memberRoles'));
    }


    public function store(Request $request)
    {

        $request->validate([
            'name' => ['required'],
            'email' => 'required|email',
            'profile_image' => ['required', 'image', 'max:3072'],
            'role_id' => ['required'],
            'country_code' => ['required'],
            'contact' => ['required'],
            'password' => ['required'],
            'unique_id' => 'required',
            'registration_number' => [
                'nullable',
                Rule::unique('member_details', 'registration_number'),
            ],
        ]);
        try {
            DB::beginTransaction();
            $data = $request->except(['_token', '_method']);

            $userData = array_intersect_key(
                $data,
                array_flip([
                    'name',
                    'email',
                    'password',
                    'country_code',
                    'contact',
                    'status',
                ])
            );

            $userDetail = array_intersect_key(
                $data,
                array_flip([
                    'dob',
                    'role_id',
                    'about',
                    'dob',
                    'extra',
                    'message',
                    'cv',
                    'registration_number'
                ])
            );
            $userData['role'] = RoleEnum::MEMBER;
            $userData["association_id"] = Auth::user()->id;
            $userData['country_code'] = "+" . $data['country_code'];
            $userData['password'] = Hash::make($userData['password']);

            $user = User::UpdateOrCreate([
                'email' => $data['email']
            ], $userData);

            if (isset($request->profile_image)) {
                $userDetail['media_id'] = $this->upload($request->profile_image)->id;
            }
            $userDetail['member_id'] = $user->id;
            $userDetail["association_id"] = Auth::user()->id;
            $userDetail["dob"] = date('Y-m-d', strtotime($userDetail["dob"]));
            $userDetail["registration_number"] = isset($userDetail["registration_number"]) ? $userDetail["registration_number"] : null;

            $user->memberDetail()->create($userDetail);

            AssociationUser::UpdateOrCreate(
                [
                    'user_id' => $user->id,
                    'association_id' => $user->association_id,
                ],
                [
                    'user_member_id' => $data['unique_id'] ?? null
                ]
            );
            DB::commit();
        } catch (\Throwable $th) {
            DB::rollBack();
            return redirect()->route('members.create')->with('error', $th->getMessage());
        }

        return redirect()->route('members.index')->with('success', 'Member Data Created');
    }

    public function edit(User $user)
    {
        $association = Auth::user();
        $memberRoles = MemberRole::where('association_id', Auth::user()->id)->where('status', StatusEnum::ACTIVE)->get();
        $countryCodes = Country::get();
        $associationUser = getAssociationUser($user, $association);
        return view('association.members.edit', compact('user', 'memberRoles', 'countryCodes', 'associationUser'));
    }

    public function update(User $user, Request $request)
    {

        $request->validate([
            'name' => ['required'],
            'email' => [
                'required',
                Rule::unique('users')->ignore($user->id),
            ],
            'profile_image' => ['nullable', 'image', 'max:3072'],
            'role_id' => ['required'],
            'country_code' => ['required'],
            'contact' => ['required'],
            'unique_id' => [
                'nullable',
                Rule::unique('users')->ignore($user->id),
            ],
            'registration_number' => [
                'nullable',
                function ($attribute, $value, $fail) use ($user) {
                    if ($value) {
                        $exists = MemberDetail::where('registration_number', $value)
                            ->where('member_id', '!=', $user->id)
                            ->exists();
                        if ($exists) {
                            $fail("The $attribute has already been taken.");
                        }
                    }
                },
            ],
        ]);

        $data = $request->except(['_token', '_method']);

        $userData = array_intersect_key(
            $data,
            array_flip([
                'name',
                'email',
                'password',
                'country_code',
                'contact',
                'status',
            ])
        );

        if ($userData['password']) {
            $userData['password'] = Hash::make($userData['password']);
        } else {
            unset($userData['password']);
        }

        $userDetail = array_intersect_key(
            $data,
            array_flip([
                'dob',
                'role_id',
                'about',
                'dob',
                'message',
                'extra',
                'cv',
                'registration_number',
            ])
        );

        $userData['country_code'] = "+" . $data['country_code'];

        $user->update($userData);

        if (isset($request->profile_image)) {
            $userDetail['media_id'] = $this->upload($request->profile_image)->id;
        }

        $userDetail["dob"] = date('Y-m-d', strtotime($userDetail["dob"]));
        $userDetail["registration_number"] = isset($userDetail["registration_number"]) ? $userDetail["registration_number"] : null;
        $user->memberDetail()->update($userDetail);

        AssociationUser::UpdateOrCreate(
            [
                'user_id' => $user->id,
                'association_id' => $user->association_id,
            ],
            [
                'user_member_id' => $data['unique_id'] ?? null
            ]
        );
        return redirect()->route('members.index', $user->type)->with('success', 'Member Data Updated');
    }


    public function statusChange(User $user)
    {
        if ($user->status == StatusEnum::ACTIVE) {
            $user->status = StatusEnum::INACTIVE;
        } else {
            $user->status = StatusEnum::ACTIVE;
        }
        $user->save();
        return response()->json(["status" => 200, "message" => "Status Changed"]);
    }


    public function destroy(User $user)
    {
        $user->delete();
        return response()->json(["status" => 200, "message" => "Member Deleted"]);
    }

    public function view(User $user)
    {
        return response()->json(["data" => $user->load('memberDetail.role')]);
    }

    public function search(Request $request)
    {
        $query = $request->input('q');

        $results = User::where('role', RoleEnum::MEMBER)
            ->where(function ($q) use ($query) {
                $q->where('name', 'LIKE', "%$query%")
                    ->orWhere('email', 'LIKE', "%$query%");
            })
            ->limit(10)
            ->get(['id', 'name', 'email']);

        return response()->json([
            'results' => $results->map(function ($item) {
                return [
                    'id' => $item->id,
                    'text' => "$item->name $item->email",
                ];
            })
        ]);
    }

    public function import(Request $request, $associationId)
    {
        $request->validate([
            'file' => ['required', 'file'],
        ]);
        try {
            Excel::import(new AssociationMemberImport($associationId), $request->file('file')->store('app'));
            return redirect()->back()->with('success', 'File imported successfully!');
        } catch (Exception $e) {
            Log::error("Error while importing file :: " .  $e->getMessage() . " on line : " . $e->getLine());
            return redirect()->back()->with('error', $e->getMessage());
        }
    }
}
