<?php

namespace App\Http\Controllers\Association;

use App\Enums\AdminLevel;
use App\Enums\StatusEnum;
use App\Helpers\Wati;
use App\Http\Controllers\Controller;
use App\Http\Traits\MediaUpload;
use App\Models\Admin;
use App\Models\Event;
use App\Models\EventManager;
use App\Models\MailTemplate;
use App\Models\MessageTemplate;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\Rule;

class EventController extends Controller
{
    use MediaUpload;

    public function index(Request $request)
    {
        $events = Event::latest()->where('association_id', $this->getCurrentUserId());
        $events = $events->filterEvent();
        $events = $events->paginate(20);
        return view('association.event.index', compact('events', 'request'));
    }

    public function create()
    {
        // $mailTemplates = MailTemplate::where('status', StatusEnum::ACTIVE)->select('id', 'name')->get();
        // $token = config('wati.token');
        // $url = config('wati.endpoint');
        // $wati = new Wati($token, $url);
        // $messageTemplates = $wati->getWatiTemplates();
        // $broadcastTemplates = isset($messageTemplates['data']) ? $messageTemplates['data'] : [];
        $eventManagers = Admin::where('association_id', $this->getCurrentUserId())->where('role', AdminLevel::EVENT_MANAGER)->whereDoesntHave('eventManager')->get();
        return view('association.event.add', compact('eventManagers'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => ['required', 'string'],
            'code' => 'required|unique:events,code',
            'banner' => ['required', 'image', 'max:3072'],
            'badge_image' => ['nullable', 'image', 'max:3072'],
            'date' => ['required'],
            'link' => ['nullable', 'url'],
            'description' => ['nullable', 'string'],
            'type' => ['required', Rule::in([Event::CONFERENCE, Event::WEBINAR, Event::OTHERS])],
            'event_manager_id' => ['nullable', 'exists:admins,id'],
        ]);
        $data = $request->except(['_token', '_method', 'banner']);
        if (isset($request->banner)) {
            $data['banner_id'] = $this->upload($request->banner)->id;
        }
        // if (isset($request->badge_image)) {
        //     $data['badge_media_id'] = $this->upload($request->badge_image)->id;
        // }
        $data["date"] = date('Y-m-d', strtotime($request->date));
        $data["association_id"] = $this->getCurrentUserId();
        $event = Event::create($data);

        if ($data['event_manager_id']) {
            EventManager::create([
                "association_id" => $this->getCurrentUserId(),
                "admin_id" => $data['event_manager_id'],
                "event_id" => $event->id,
            ]);
        }
        return redirect()->route('events.index')->with('success', 'Event Created Successfully.');
    }

    public function edit(Event $event)
    {
        // $messageTemplates = MessageTemplate::where('status', MessageTemplate::APPROVED)->select('id', 'name')->get();
        // $mailTemplates = MailTemplate::where('status', StatusEnum::ACTIVE)->select('id', 'name')->get();
        // $token = config('wati.token');
        // $url = config('wati.endpoint');
        // $wati = new Wati($token, $url);
        // $messageTemplates = $wati->getWatiTemplates();
        // $broadcastTemplates = isset($messageTemplates['data']) ? $messageTemplates['data'] : [];
        $selectedManager = EventManager::where('event_id', $event->id)->first();
        $eventManagers = Admin::where('association_id', $this->getCurrentUserId())
            ->where('role', AdminLevel::EVENT_MANAGER)
            ->whereDoesntHave('eventManager')
            ->orWhere(function ($query) use ($selectedManager) {
                if ($selectedManager) {
                    $query->where('id', $selectedManager->admin_id);
                }
            })
            ->get();
        return view('association.event.edit', compact('event', 'eventManagers', 'selectedManager'));
    }

    public function update(Event $event, Request $request)
    {
        $request->validate([
            'title' => ['required', 'string'],
            'code' => [
                'required',
                Rule::unique('events')->ignore($event->id),
            ],
            'banner' => ['nullable', 'image', 'max:3072'],
            'date' => ['required'],
            'link' => ['nullable', 'url'],
            'description' => ['nullable', 'string'],
            'type' => ['required', Rule::in([Event::CONFERENCE, Event::WEBINAR, Event::OTHERS])],
            'event_manager_id' => ['nullable', 'exists:admins,id'],
        ]);
        $data = $request->except(['_token', '_method']);
        if (isset($request->banner)) {
            $data['banner_id'] = $this->upload($request->banner)->id;
        }
        // if (isset($request->badge_image)) {
        //     $data['badge_media_id'] = $this->upload($request->badge_image)->id;
        // }
        $data["date"] = date('Y-m-d', strtotime($request->date));
        $event->update($data);

        if ($data['event_manager_id']) {
            EventManager::updateOrCreate(
                ['event_id' => $event->id],
                [
                    'association_id' => $this->getCurrentUserId(),
                    'admin_id' => $data['event_manager_id'],
                    'event_id' => $event->id,
                ]
            );
        } else {
            EventManager::where('event_id', $event->id)->delete();
        }

        return redirect()->route('events.index')->with('success', 'Event Updated Successfully.');
    }

    public function statusChange(Event $event)
    {
        $newStatus = ($event->status == StatusEnum::ACTIVE) ? StatusEnum::INACTIVE : StatusEnum::ACTIVE;
        $event->status = $newStatus;
        $event->save();
        return response()->json(["status" => 200, "message" => "Status Changed Successfully."]);
    }

    public function destroy(Event $event)
    {
        $event->delete();
        return response()->json(["status" => 200, "message" => "Event Deleted Successfully."]);
    }
}
