<?php

namespace App\Http\Controllers\Association;

use App\Enums\StatusEnum;
use App\Http\Controllers\Controller;
use App\Http\Traits\MediaUpload;
use App\Models\Committee;
use App\Models\CommitteeMember;
use App\Models\MemberRole;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class CommitteeController extends Controller
{
    use MediaUpload;

    public function index(Request $request)
    {

        $committees = Committee::where('association_id', Auth::user()->id);

        $committees = $committees->paginate(25);

        return view('association.committees.index', compact('committees', 'request'));
    }

    public function create()
    {
        return view('association.committees.add');
    }


    public function store(Request $request)
    {
        $request->validate([
            'name' => ['required'],
        ]);

        $data = $request->except(['_token', '_method']);
        $data["association_id"] = Auth::user()->id;
        Committee::create($data);

        return redirect()->route('committees.index')->with('success', 'Committe Created');
    }

    public function edit(Committee $committee)
    {
        return view('association.committees.edit', compact('committee'));
    }

    public function update(Committee $committee, Request $request)
    {
        $request->validate([
            'name' => ['required'],
        ]);

        $data = $request->except(['_token', '_method']);
        $committee->update($data);
        return redirect()->route('committees.index')->with('success', 'Committee Updated');
    }

    public function updateCommiteeIndex(Request $request, Committee $committee)
    {
        $request->validate([
            'index' => 'required',
        ]);
        try {
            $committee->index = $request->index;
            $committee->update();

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => 'success',
                    'message' => 'Index change successfully',
                ]);
            }
            return redirect()->route('international.affiliations.index');
        } catch (\Exception $e) {
            $this->logError($e, "Update index international affiliations");
            return redirect()->back()->with(['error', $e->getMessage()]);
        }
    }

    public function view(Committee $committee, Request $request)
    {
        $members = User::whereHas('memberDetail', function ($q) use ($request) {
            $q->where('association_id', Auth::user()->id);
        })
            ->where('status', StatusEnum::ACTIVE)
            ->get();

        $committeeMembers = CommitteeMember::where("commity_id", $committee->id)->whereHas('member');

        if ($request->role_id) {
            $committeeMembers->whereHas('member', function ($query) use ($request) {
                $query->whereHas('memberDetail', function ($query) use ($request) {
                    if ($request->role_id != 'all') {
                        $query->where('role_id', $request->role_id);
                    }
                });
            });
        }
        if ($request->name) {
            $committeeMembers->whereHas('member', function ($query) use ($request) {
                $query->where("name", 'like', '%' . $request->name . '%');
            });
        }
        if ($request->contact) {
            $committeeMembers->whereHas('member', function ($query) use ($request) {
                $query->where("contact", 'like', '%' . $request->contact . '%');
            });
        }

        if (!empty($request->status) && $request->status != 'all') {
            if ($request->status == "inactive") {
                $statusData = StatusEnum::INACTIVE;
            }
            if ($request->status == "active") {
                $statusData = StatusEnum::ACTIVE;
            }
            $members->where('status', $statusData);
        }

        $committeeMembers = $committeeMembers->orderBy('index')->get();

        if ($request->export == "export") {

            $fileName = $committee->name . ".csv";
            $headers = [
                "Content-type"        => "text/csv",
                "Content-Disposition" => "attachment; filename=$fileName",
                "Pragma"              => "no-cache",
                "Cache-Control"       => "must-revalidate, post-check=0, pre-check=0",
                "Expires"             => "0"
            ];

            $columns = ['Sr No.', 'Name', 'Contact', 'Role', 'About', 'Date'];

            // Callback function for CSV generation
            $callback = function () use ($committeeMembers, $columns) {
                $file = fopen('php://output', 'w');
                fputcsv($file, $columns);
                $i = 1;  // Start from 1 to begin numbering from 2nd row
                foreach ($committeeMembers as $committeeMember) {
                    fputcsv($file, [
                        $i,
                        $committeeMember->member->name ?? '-',
                        $committeeMember->member->contact ?? '-',
                        $committeeMember->member->memberDetail->role->name ?? '-',
                        $committeeMember->member->memberDetail->about ?? '-',
                        $committeeMember->member->created_at ? $committeeMember->member->created_at->timezone('Asia/Kolkata')->format('d-m-Y | h:i A') : '',
                    ]);
                    $i++;
                }
                fclose($file);
            };

            return response()->stream($callback, 200, $headers);
        }


        $memberRoles = MemberRole::where('association_id', Auth::user()->id)->get();

        return view('association.committees.view', compact('committee', 'members', 'committeeMembers', 'memberRoles', 'request'));
    }

    public function createMember(Committee $committee, Request $request)
    {
        $users = [];

        if (count($request->all()) > 0) {
            $users = User::where('status', StatusEnum::ACTIVE)
                ->whereDoesntHave('committeeMember', function ($q) use ($committee) {
                    $q->where('commity_id', $committee->id);
                })
                ->whereHas('associationUsers', function ($q) {
                    $q->where('association_id', Auth::user()->id);
                });

            if (!empty($request->name) && $request->name != 'all') {
                $users->where('name', 'like', "%" . $request->name . "%");
            }

            if ($request->role_id) {
                $users->whereHas('memberDetail', function ($query) use ($request) {
                    if ($request->role_id != 'all') {
                        $query->where('role_id', $request->role_id);
                    }
                });
            }

            if (!empty($request->mobile) && $request->mobile != 'all') {
                $users->where('contact', 'like', "%" . $request->mobile . "%");
            }

            if (!empty($request->status) && $request->status != 'all') {
                if ($request->status == "inactive") {
                    $statusData = StatusEnum::INACTIVE;
                }
                if ($request->status == "active") {
                    $statusData = StatusEnum::ACTIVE;
                }
                $users->where('status', $statusData);
            }

            $users = $users->get();
        }
        $memberRoles = MemberRole::where('association_id', Auth::user()->id)->get();

        return view('association.committees.add-members', compact('committee', 'users', 'request', 'memberRoles'));
    }


    public function addMember(Committee $committee, Request $request)
    {
        $request->validate([
            'member_ids' => ['required'],
        ]);

        $data = $request->except(['_token', '_method']);

        foreach ($data['member_ids'] as $memberId) {
            $user = CommitteeMember::where('commity_id', $committee->id)->where('member_id', $memberId)->first();
            if (!$user) {
                CommitteeMember::create([
                    'association_id' => Auth::user()->id,
                    'commity_id' => $committee->id,
                    'member_id' => $memberId,
                    'index' => CommitteeMember::max('index') + 1,
                ]);
            }
        }

        return redirect()->route('committees.view', $committee->id)->with('success', 'Committe Member Added');
    }

    public function destroyMember(CommitteeMember $committeeMember)
    {
        $committeeMember->delete();
        return response()->json(["status" => 200, "message" => "Member Deleted"]);
    }

    public function statusChange(Committee $committee)
    {
        if ($committee->status == StatusEnum::ACTIVE) {
            $committee->status = StatusEnum::INACTIVE;
        } else {
            $committee->status = StatusEnum::ACTIVE;
        }
        $committee->save();
        return response()->json(["status" => 200, "message" => "Status Changed"]);
    }

    public function destroy(Committee $committee)
    {
        $committee->delete();
        return response()->json(["status" => 200, "message" => "Committee Deleted"]);
    }

    public function updateIndex(Request $request, CommitteeMember $committeeMember)
    {
        $request->validate([
            'index' => 'required',
        ]);
        try {
            $committeeMember->index = $request->index;
            $committeeMember->update();

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => 'success',
                    'message' => 'Index change successfully',
                ]);
            }
            return redirect()->route('committees.view', $committeeMember->id);
        } catch (\Exception $e) {
            return redirect()->back()->with(['error', $e->getMessage()]);
        }
    }
}
