<?php

namespace App\Http\Controllers\Association;

use App\Enums\StatusEnum;
use App\Http\Controllers\Controller;
use App\Http\Traits\MediaUpload;
use App\Models\Banner;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class BannerController extends Controller
{
    use MediaUpload;

    public function index(Request $request)
    {
        $banners = Banner::where('association_id', Auth::user()->id)->orderBy('index');

        $banners = $banners->paginate(25);

        return view('association.banners.index', compact('banners', 'request'));
    }

    public function create()
    {
        return view('association.banners.add');
    }

    public function store(Request $request)
    {
        $request->validate([
            'banner_image' => ['required', 'image', 'max:3072'],
        ]);

        $data = $request->except(['_token', '_method']);

        if (isset($request->banner_image)) {
            $data['media_id'] = $this->upload($request->banner_image)->id;
        }
        $data["association_id"] = Auth::user()->id;
        $data["index"] = $request->index ?? Banner::max('index') + 1;

        Banner::create($data);

        return redirect()->route('banners.index')->with('success', 'Banner Created');
    }

    public function edit(Banner $banner)
    {
        return view('association.banners.edit', compact('banner'));
    }

    public function update(Banner $banner, Request $request)
    {
        $request->validate([
            'banner_image' => ['nullable', 'image', 'max:3072'],
        ]);

        $data = $request->except(['_token', '_method']);
        if (isset($request->banner_image)) {
            $this->updateMedia($banner->media_id, $request->banner_image, $banner->image->path);
        }
        $data["index"] = $request->index ?? Banner::max('index') + 1;
        $banner->update($data);
        return redirect()->route('banners.index')->with('success', 'Banner Updated');
    }

    public function statusChange(Banner $banner)
    {
        if ($banner->status == StatusEnum::ACTIVE) {
            $banner->status = StatusEnum::INACTIVE;
        } else {
            $banner->status = StatusEnum::ACTIVE;
        }
        $banner->save();
        return response()->json(["status" => 200, "message" => "Banner Status Changed"]);
    }

    public function destroy(Banner $banner)
    {
        $banner->delete();
        return response()->json(["status" => 200, "message" => "Banner Deleted"]);
    }
}
