<?php

namespace App\Http\Controllers\Admin;

use App\Enums\MessageCampaignEnum;
use App\Enums\RoleEnum;
use App\Enums\StatusEnum;
use App\Helpers\MetaClient;
use App\Http\Controllers\Controller;
use App\Http\Traits\MediaUpload;
use App\Jobs\Meta\CreateTemplateJob;
use App\Models\Committee;
use App\Models\Event;
use App\Models\MemberRole;
use App\Models\MessageCampaign;
use App\Models\MessageCampaignUser;
use App\Models\MessageTemplate;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class MessageTemplateController extends Controller
{
    use MediaUpload;

    public function index(Request $request)
    {
        $messageTemplates = MessageTemplate::orderBy('id', 'DESC');
        if (isset($request->name) && !empty($request->name)) {
            $messageTemplates->where('name', 'like', "%" . $request->name . "%");
        }
        $messageTemplates = $messageTemplates->paginate(50);
        return view('admin.message-templates.index', compact('messageTemplates', 'request'));
    }

    public function create()
    {
        $columnsDisplay = [
            'name',
            'role_name',
            'committee_name'
        ];
        return view('admin.message-templates.create', compact('columnsDisplay'));
    }

    public function save(Request $request)
    {
        $request->validate([
            "template_name" => "required",
            "language" => "required",
            "category" => "required",

            "header_type" => "nullable|in:None,TEXT,MEDIA",
            "header_text" => "required_if:header_type,TEXT",
            "media_file" => "required_if:header_type,MEDIA",
            "header_dynamic_value" => "nullable|array",

            "body_text" => "required",
            "body_dynamic_value" => "nullable|array",

            "footer_text" => "nullable",

            "buttons" => "nullable|array",
            "buttons.*.type" => "nullable|in:PHONE_NUMBER,URL,QUICK_REPLY",
            "buttons.*.text" => "nullable|string",
            "buttons.*.value" => "nullable|string",
        ]);

        try {
            DB::beginTransaction();

            $messageTemplate = new MessageTemplate($request->all());
            $messageTemplate->name = $request->get("template_name");
            $messageTemplate->allow_category_change = 0;
            $messageTemplate->status = MessageTemplate::PENDING;
            $messageTemplate->save();

            // header part
            if ($request->get("header_type") && $request->header_type != "None") {
                $mediaFile = null;
                if (isset($request->media_file)) {
                    $mediaPath = $this->upload($request->media_file)->id;
                    $mediaFile = $mediaPath;
                }
                $headerComponent = [
                    'type' => 'HEADER',
                    'format' => $request->get("header_type") == "MEDIA" ? strtoupper($request->media_type) : $request->get("header_type"),
                    'text' => $request->get("header_type") == "TEXT" ? $request->get("header_text") : null,
                    "example_key" => "header_text",
                    "media_id" => $request->get("header_type") == "MEDIA" ? $mediaFile : null,
                    "dynamic_variables" => $request->get("header_type") == "TEXT" ? $request->get("header_dynamic_value") : null
                ];
                $messageTemplate->components()->create($headerComponent);
            }

            // body part
            if ($request->get("body_text")) {
                $bodyComponent = [
                    "type" => "BODY",
                    "text" => $request->get("body_text"),
                    "example_key" => "body_text",
                    "dynamic_variables" => $request->get("body_dynamic_value"),
                ];
                $messageTemplate->components()->create($bodyComponent);
            }

            // footer part
            if ($request->get("footer_text")) {
                $messageTemplate->components()->create([
                    "type" => "FOOTER",
                    "text" => $request->footer_text,
                ]);
            }

            $buttons = $request->get("buttons", []);
            foreach ($buttons as $key => $button) {
                $value = $button["value"] ?? "";
                if (isset($button["type"]) && $button['type'] == 'PHONE_NUMBER') {
                    $country = $button["country"] ?? "";
                    $value = $country . $value;
                }
                $buttonComponent = [
                    "type" => $button["type"] ?? "",
                    "format" => $value,
                    "text" => $button["text"] ?? "",
                ];
                $messageTemplate->components()
                    ->create($buttonComponent);
            }

            DB::commit();

            CreateTemplateJob::dispatchSync($messageTemplate->id);

            return response([
                'status' => true,
                'messages' => ['Template saved, Waiting for approval']
            ]);
        } catch (\Exception $exception) {
            DB::rollBack();
            return response([
                'status' => false,
                'messages' => $exception->getMessage(),
            ]);
        }
    }

    public function retry(MessageTemplate $messageTemplate, Request $request)
    {
        if ($messageTemplate->status != MessageTemplate::FAILED) {
            return redirect()->back()->with('error', 'Your template is not failed');
        }
        $messageTemplate->status = MessageTemplate::PENDING;
        $messageTemplate->save();
        CreateTemplateJob::dispatchSync($messageTemplate->id);

        return redirect()->back()->with('success', 'Wait For Template Approval');
    }

    public function show(MessageTemplate $messageTemplate)
    {
        return view('admin.message-templates.show', compact('messageTemplate'));
    }

    public function destroy(MessageTemplate $messageTemplate)
    {
        $response = MetaClient::api()->deleteMessageTemplate($messageTemplate->template_id, $messageTemplate->name);
        if ($response->ok()) {
            $messageTemplate->delete();
            return response()->json([
                "status" => 200,
                "message" => "Message Template Deleted"
            ]);
        } else {
            $data = $response->json();
            if (!empty($data) && !empty($data['error'])) {
                return response()->json(["status" => 400, "message" => $data['error']['error_user_msg']]);
            }
            return response()->json(["status" => 400, "message" => 'Something went wrong']);
        }
    }

    // Bulk messages
    public function messageCampaigns()
    {
        $messageCampaigns = MessageCampaign::orderBy('id', 'DESC')->paginate(100);
        return view('admin.message-campaigns.index', compact('messageCampaigns'));
    }

    public function messageCampaignUsers(MessageCampaign $messageCampaign, Request $request)
    {
        $messageCampaignUsers = MessageCampaignUser::where('message_campaign_id', $messageCampaign->id);
        // Export undelivered message
        if ($request->export) {
            $messageCampaignUsers = $messageCampaignUsers->where('status', MessageCampaignUser::STATUS_SENT)->whereNull('delivered_at')->get();
            $fileName = $messageCampaign->template_name . "_undelivered_message.csv";
            $headers = array(
                "Content-type" => "text/csv",
                "Content-Disposition" => "attachment; filename=$fileName",
                "Pragma" => "no-cache",
                "Cache-Control" => "must-revalidate, post-check=0, pre-check=0",
                "Expires" => "0"
            );
            $columns = array(
                'Sr No.',
                'Name',
                'Mobile Number'
            );
            $callback = function () use ($messageCampaignUsers, $columns) {
                $file = fopen('php://output', 'w');
                fputcsv($file, $columns);
                $i = 1;
                foreach ($messageCampaignUsers as $messageCampaignUser) {
                    $row['sr_no'] = $i;
                    $row['name'] = $messageCampaignUser->user->name;
                    $row['mobile_number'] = $messageCampaignUser->user->country_code . " " . $messageCampaignUser->user->mobile;
                    fputcsv($file, array(
                        $row['sr_no'],
                        $row['name'],
                        $row['mobile_number']
                    ));
                    $i++;
                }
                fclose($file);
            };
            return response()->stream($callback, 200, $headers);
        }
        $messageCampaignUsers = $messageCampaignUsers->get();
        return view('admin.message-campaigns.users', compact('messageCampaignUsers', 'messageCampaign'));
    }

    public function bulkMessages(Request $request)
    {
        $renewalDate = null;
        $users = [];
        $participants = [];
        if (count($request->all()) > 0) {
            $users = User::where('role', RoleEnum::MEMBER);

            if (!empty($request->name) && $request->name != 'all') {
                $users->where('name', 'like', "%" . $request->name . "%");
            }

            if (!empty($request->mobile) && $request->mobile != 'all') {
                $users->where('contact', 'like', "%" . $request->mobile . "%");
            }

            if (!empty($request->status) && $request->status != 'all') {
                if ($request->status == "inactive") {
                    $statusData = StatusEnum::INACTIVE;
                }
                if ($request->status == "active") {
                    $statusData = StatusEnum::ACTIVE;
                }
                $users->where('status', $statusData);
            }

            if (!empty($request->association_id) && $request->association_id != 'all') {
                $users->whereHas('memberDetail', function ($q) use ($request) {
                    $q->where('association_id', $request->association_id);
                });
            }

            if (!empty($request->committee_id) && $request->committee_id != 'all') {
                $users->whereHas('memberDetail', function ($q) use ($request) {
                    $q->join('committee_members', 'member_details.member_id', '=', 'committee_members.member_id')
                        ->where('committee_members.commity_id', $request->committee_id)
                        ->select('member_details.*');
                });
            }
            $users = $users->get();

            if (!empty($request->event_id) && $request->event_id != 'all') {
                $participants = User::whereHas('eventRegistrations', function ($q) use ($request) {
                    $q->where('event_id', $request->event_id);
                })->get();
            }
        }

        $committees = Committee::where('status', StatusEnum::ACTIVE)->orderBy('id', 'desc')->get();
        $memberRoles = MemberRole::where('status', StatusEnum::ACTIVE)->orderBy('id', 'desc')->get();
        $messageTemplates = MessageTemplate::where('status', MessageTemplate::APPROVED)->select('id', 'name')->get();
        $associations = User::where('role', RoleEnum::ASSOCIATION)->where('status', StatusEnum::ACTIVE)->orderBy('id', 'desc')->get();
        $events = Event::where('status', StatusEnum::ACTIVE)->get();
        return view('admin.message-campaigns.bulk-message', compact('messageTemplates', 'request', 'users', 'committees', 'memberRoles', 'associations', 'events', 'participants'));
    }

    public function sendBulkMessages(Request $request)
    {
        if (isset($request->mobile)) {
            $countryCode = "+" . $request->country_code;
            $users = User::where('country_code', $countryCode)->where('contact', $request->mobile)->pluck('id')->toArray();
            if (count($users) == 0) {
                return redirect()->back()->with('error', 'User is not available');
            }
            $request->merge([
                'users' => $users,
            ]);

        }

        $template = MessageTemplate::find($request->template_name);
        if (!$template) {
            return redirect()->back()->with('error', 'Template is not available');
        }

        if (!$request->users) {
            return redirect()->back()->with('error', 'User is not selected');
        }

        $usersArray = array_unique($request->users);

        if (!is_array($usersArray)) {
            $request->merge([
                'users' => json_decode($usersArray),
            ]);
        }

        if (!is_array($request->users)) {
            Log::info("Message Users: " . json_encode($request->users));
            return redirect()->back()->with('error', 'Something went wrong Please Try Again.');
        }
        $scheduledDate = date('Y-m-d');
        $scheduledTime = date('H:i');

        if (isset($request->schedule_type) && $request->schedule_type != 'send_now') {
            $scheduledDate = date("Y-m-d", strtotime($request->schedule_date));
            $scheduledTime = $request->schedule_time;
        }
        $scheduledType = !empty($request->schedule_type) ? $request->schedule_type : 'send_now';

        $messageCampaign = new MessageCampaign();
        $messageCampaign->template_name = $template->name;
        $messageCampaign->scheduled_date = $scheduledDate;
        $messageCampaign->scheduled_time = $scheduledTime;
        $messageCampaign->total_users = count($usersArray);
        $messageCampaign->users = $usersArray;
        $messageCampaign->status = MessageCampaignEnum::PENDING;
        $messageCampaign->schedule_type = $scheduledType;
        $messageCampaign->message_template_id = $template->id;
        $messageCampaign->scheduled_at = $scheduledDate . " " . $scheduledTime;
        $messageCampaign->save();

        if ($scheduledType == 'send_now') {
            $host = $request->getHost();
            $messageCampaign->start($host);
            return redirect()->route('bulk.messages.campaigns')->with('success', 'Message send Successfully');
        } else {
            return redirect()->route('bulk.messages.campaigns')->with('success', 'Message scheduled Successfully');
        }
    }

    public function getCommittees($association_id)
    {
        $committees = Committee::where('association_id', $association_id)->where('status', StatusEnum::ACTIVE)->get();
        return response()->json(["status" => 200, "data" => $committees, "message" => "Committee fetch successfully"]);
    }

    public function getEvents($association_id)
    {
        $events = Event::where('association_id', $association_id)->where('status', StatusEnum::ACTIVE)->get();
        return response()->json(["status" => 200, "data" => $events, "message" => "Event fetch successfully"]);
    }
}
