<?php

namespace App\Http\Controllers\Admin;

use App\Enums\MailTemplateCampaignEnum;
use App\Enums\RoleEnum;
use App\Enums\StatusEnum;
use App\Helpers\SendEmailCampaign;
use App\Http\Controllers\Controller;
use App\Http\Traits\MediaUpload;
use App\Models\Committee;
use App\Models\Event;
use App\Models\MailTemplate;
use App\Models\MailTemplateCampaign;
use App\Models\MemberRole;
use App\Models\User;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class MailTemplateController extends Controller
{
    use MediaUpload;

    public function index(Request $request)
    {
        $mailTemplates = MailTemplate::latest();

        if ($request->name) {
            $mailTemplates->where("name", 'like', '%' . $request->name . '%');
        }
        if ($request->template_id) {
            $mailTemplates->where("template_id", 'like', '%' . $request->template_id . '%');
        }
        if ($request->subject) {
            $mailTemplates->where("subject", 'like', '%' . $request->subject . '%');
        }

        $mailTemplates = $mailTemplates->paginate(25);

        return view('admin.mail-templates.index', compact('mailTemplates', 'request'));
    }

    public function create()
    {
        $columnsDisplay = [
            'name',
            'contact',
        ];
        $mailjetTemplates = getMailjetTemplates();
        return view('admin.mail-templates.create', compact('columnsDisplay', 'mailjetTemplates'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => ['required'],
            'template_id' => ['required'],
            'from' => ['required'],
            'subject' => ['required'],
        ]);
        $data = $request->except(['_method', '_token']);
        $attachmentArray = [];
        if ($request->attachments && $request->attachments > 0) {
            foreach ($request->attachments as $attachment) {
                $size = $attachment->getSize() / 1024 / 1024;
                if ($size <= 2) {
                    $media = $this->upload($attachment)->id;
                    $attachmentArray[] = $media;
                }
            }
            $data['attachment'] = $attachmentArray;
        }
        MailTemplate::create($data);
        return redirect()->route('mail.templates.index')->with('success', 'Email Template Created');
    }

    public function edit(MailTemplate $mailTemplate)
    {
        $columnsDisplay = [
            'name',
            'contact',
        ];
        $mailjetTemplates = getMailjetTemplates();
        return view('admin.mail-templates.edit', compact('mailTemplate', 'columnsDisplay', 'mailjetTemplates'));
    }

    public function update(MailTemplate $mailTemplate, Request $request)
    {
        $data = $request->except(['_method', '_token']);
        $attachmentArray = [];
        if ($request->attachments && $request->attachments > 0) {
            foreach ($request->attachments as $attachment) {
                $size = $attachment->getSize() / 1024 / 1024;
                if ($size <= 2) {
                    $media = $this->upload($attachment)->id;
                    $attachmentArray[] = $media;
                }
            }
            $data['attachment'] = $attachmentArray;
            if ($mailTemplate->getAttachments()) {
                foreach ($mailTemplate->getAttachments() as $attachment) {
                    if ($attachment && $attachment->path) {
                        $this->remove($attachment->path);
                    }
                }
            }
        }
        $mailTemplate->update($data);
        return redirect()->route('mail.templates.index')->with('success', 'Email Template Updated');
    }

    public function statusChange(MailTemplate $mailTemplate)
    {
        if ($mailTemplate->status == StatusEnum::ACTIVE) {
            $mailTemplate->status = StatusEnum::INACTIVE;
        } else {
            $mailTemplate->status = StatusEnum::ACTIVE;
        }
        $mailTemplate->save();
        return response()->json(["status" => 200, "message" => "Status Changed"]);
    }

    public function destroy(MailTemplate $mailTemplate)
    {
        if ($mailTemplate->getAttachments()) {
            foreach ($mailTemplate->getAttachments() as $attachment) {
                if ($attachment && $attachment->path) {
                    $this->remove($attachment->path);
                    $attachment->delete();
                }
            }
        }
        $mailTemplate->delete();
        return response()->json(["status" => 200, "message" => "Mail Template Deleted"]);
    }

    // Bulk email functions
    public function bulkEmails()
    {
        $mailTemplates = MailTemplateCampaign::latest()->paginate(10);
        return view('admin.bulk-emails.bulk-email', compact('mailTemplates'));
    }

    public function sendEmails(Request $request)
    {
        $renewalDate = null;
        $users = [];
        $participants = [];
        if (count($request->all()) > 0) {
            $users = User::where('role', RoleEnum::MEMBER);

            if (!empty($request->name) && $request->name != 'all') {
                $users->where('name', 'like', "%" . $request->name . "%");
            }

            if (!empty($request->mobile) && $request->mobile != 'all') {
                $users->where('contact', 'like', "%" . $request->mobile . "%");
            }

            if (!empty($request->status) && $request->status != 'all') {
                if ($request->status == "inactive") {
                    $statusData = StatusEnum::INACTIVE;
                }
                if ($request->status == "active") {
                    $statusData = StatusEnum::ACTIVE;
                }
                $users->where('status', $statusData);
            }

            if (!empty($request->association_id) && $request->association_id != 'all') {
                $users->whereHas('memberDetail', function ($q) use ($request) {
                    $q->where('association_id', $request->association_id);
                });
            }

            if (!empty($request->committee_id) && $request->committee_id != 'all') {
                $users->whereHas('memberDetail', function ($q) use ($request) {
                    $q->join('committee_members', 'member_details.member_id', '=', 'committee_members.member_id')
                        ->where('committee_members.commity_id', $request->committee_id)
                        ->select('member_details.*');
                });
            }

            $users = $users->get();

            if (!empty($request->event_id) && $request->event_id != 'all') {
                $participants = User::whereHas('eventRegistrations', function ($q) use ($request) {
                    $q->where('event_id', $request->event_id);
                })->get();
            }
        }
        $committees = Committee::where('status', StatusEnum::ACTIVE)->orderBy('id', 'desc')->get();
        $memberRoles = MemberRole::where('status', StatusEnum::ACTIVE)->orderBy('id', 'desc')->get();
        $mailTemplates = MailTemplate::where('status', StatusEnum::ACTIVE)->select('id', 'name')->get();
        $associations = User::where('role', RoleEnum::ASSOCIATION)->where('status', StatusEnum::ACTIVE)->orderBy('id', 'desc')->get();
        $events = Event::where('status', StatusEnum::ACTIVE)->get();
        return view('admin.bulk-emails.send-emails', compact('mailTemplates', 'request', 'users', 'committees', 'memberRoles', 'associations', 'events', 'participants'));
    }

    public function sendBulkEmails(Request $request)
    {
        if (!$request->users) {
            return redirect()->back()->with('error', 'Please Select Users');
        }

        $usersArray = array_unique($request->users);

        if (!is_array($request->users)) {
            $usersArray = json_decode($request->users);
        }

        if (is_array($usersArray) && count($usersArray) <= 0) {
            return redirect()->back()->with('error', 'Please Select Users');
        }

        try {

            $mailTemplateCampaign = new MailTemplateCampaign();
            $mailTemplateCampaign->template_name = $request->template_name;

            if ($request->schedule_type != 'send_now') {
                $mailTemplateCampaign->scheduled_date = date('Y-m-d', strtotime($request->schedule_date));
                $mailTemplateCampaign->scheduled_time = date('H:i', strtotime($request->schedule_time));
            } else {
                $mailTemplateCampaign->scheduled_date = date('Y-m-d');
                $mailTemplateCampaign->scheduled_time = date('H:i');
            }

            $mailTemplateCampaign->total_users = count($usersArray);
            $mailTemplateCampaign->users = $usersArray;
            $mailTemplateCampaign->success_users = [];
            $mailTemplateCampaign->fail_users = [];
            $mailTemplateCampaign->status = MailTemplateCampaignEnum::PENDING;
            $mailTemplateCampaign->schedule_type = $request->schedule_type;
            $mailTemplateCampaign->save();
            $host = $request->getHost();

            if ($request->schedule_type == 'send_now') {
                (new SendEmailCampaign())->sendEmailCampaign($host, $mailTemplateCampaign);
                return redirect()->back()->with('success', 'Emails are sending now.');
            } else {
                return redirect()->back()->with('success', 'Email scheduled Successfully');
            }
        } catch (Exception $e) {
            Log::error("Bulk Email Error -" . $e->getMessage());
            return redirect()->back()->with('error', 'Something Went Wrong Please Try Again Later');
        }
    }
}
