<?php

namespace App\Http\Controllers\API;

use App\Enums\StatusEnum;
use App\Http\Controllers\Controller;
use App\Http\Resources\API\PollResource;
use App\Models\Poll;
use App\Models\PollOption;
use Illuminate\Http\Request;

class PollController extends Controller
{
    public function index(Request $request)
    {
        try {
            $userId = $request->user_id ?? null;

            date_default_timezone_set('Asia/Kolkata');
            $currentDate = date('Y-m-d');
            $currentTime = date('H:i:s');

            $polls = Poll::where('association_id', $request->attributes->get('association_id'))
                ->where('status', StatusEnum::ACTIVE)
                ->where(function ($query) use ($currentDate, $currentTime) {
                    $query->where(function ($q) use ($currentDate, $currentTime) {
                        $q->where('start_date', '<', $currentDate)
                            ->orWhere(function ($subQ) use ($currentDate, $currentTime) {
                                $subQ->where('start_date', '=', $currentDate)
                                    ->where('start_time', '<=', $currentTime);
                            });
                    })
                        ->where(function ($q) use ($currentDate, $currentTime) {
                            $q->where('expire_date', '>', $currentDate)
                                ->orWhere(function ($subQ) use ($currentDate, $currentTime) {
                                    $subQ->where('expire_date', '=', $currentDate)
                                        ->where('expire_time', '>', $currentTime);
                                });
                        });
                })
                ->get();
            if($userId){
                $request->merge(['auth_user' => true, 'user_id' => $userId]);
            }
            return PollResource::collection($polls);
        } catch (\Exception $e) {
            $this->logError($e, "Get Polls Error");
            return $this->sendError($e->getMessage());
        }
    }

    public function addVote(Request $request, PollOption $pollOption)
    {
        try {
            $user = $request->user();
            $poll = $pollOption->poll;
            $existingVote = $poll->pollOptions()->whereHas('votedByUsers', function($query) use ($user) {
                $query->where('user_id', $user->id);
            })->first();

            if ($existingVote) {
                $existingVote->removeVote($user);
                $existingVote->updateVoteCount();
            }
            
            $pollOption->VotePoll($user);
            $pollOption->updateVoteCount();
            $message = 'Thank you for submitting';

            return response()->json([
                'message' => $message
            ]);
        } catch (\Exception $e) {
            $this->logError($e, "Poll Vote Error");
            return $this->sendError($e->getMessage());
        }
    }
}
