<?php

namespace App\Http\Controllers\API;

use App\Enums\StatusEnum;
use App\Http\Controllers\Controller;
use App\Http\Resources\API\MemberResource;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

class MemberController extends Controller
{
    public function index(Request $request)
    {
        try {
            $members = User::whereHas('memberDetail', function ($q) use ($request) {
                $q->where('association_id', $request->attributes->get('association_id'));
            })->where('status', StatusEnum::ACTIVE)
                ->latest();

            if ($request->name) {
                $members = $members->where('name', 'like', '%' . $request->name . '%');
            }

            $members = $members->simplePaginate(20);

            return MemberResource::collection($members);
        } catch (\Exception $e) {
            $this->logError($e, "Get Members Error");
            return $this->sendError($e->getMessage());
        }
    }

    public function upcomingBirthday(Request $request)
    {
        try {
            $currentDate = now();
            $currentMonth = $currentDate->month;
            $currentDay = $currentDate->day;

            $members = User::whereHas('memberDetail', function ($q) use ($request) {
                $q->where('association_id', $request->attributes->get('association_id'))->whereNotNull('dob');
            })
                ->where('status', StatusEnum::ACTIVE)
                ->get()
                ->filter(function ($member) use ($currentMonth, $currentDay) {
                    $dob = \Carbon\Carbon::parse($member->memberDetail->dob);
                    $upcomingBirthdayMonth = $dob->month;
                    $upcomingBirthdayDay = $dob->day;

                    if (
                        ($upcomingBirthdayMonth == $currentMonth && $upcomingBirthdayDay >= $currentDay) ||
                        ($upcomingBirthdayMonth > $currentMonth)
                    ) {
                        return true;
                    }

                    return false;
                });
            $data = MemberResource::collection($members);
            return $this->sendJson($data);
        } catch (\Exception $e) {
            $this->logError($e, "Get member upcoming birthday Error");
            return $this->sendError($e->getMessage());
        }
    }

    public function message(Request $request)
    {
        try {
            $roleOrders = [
                6098 => ['Chairman', 'Vice Chairman', 'Secretary', 'Past Chairman'],
                'default' => ['President', 'Secretary', 'Vice President'],
            ];

            $associationId = $request->attributes->get('association_id');
            $customRoleOrder = $roleOrders[$associationId] ?? $roleOrders['default'];

            $members = User::whereHas('memberDetail', function ($q) use ($request, $customRoleOrder) {
                $q->with('role')
                    ->where('association_id', $request->attributes->get('association_id'))
                    ->whereNotNull('dob')
                    ->whereHas('role', function ($query) use ($customRoleOrder) {
                        $query->whereIn('name', $customRoleOrder);
                    });
            })->where('status', StatusEnum::ACTIVE)
                ->with(['memberDetail.role'])
                ->get();

            // Sort collection by custom role order
            $sortedMembers = $members->sortBy(function ($member) use ($customRoleOrder) {
                $roleName = $member->memberDetail->role->name ?? '';
                return array_search($roleName, $customRoleOrder) !== false
                    ? array_search($roleName, $customRoleOrder)
                    : PHP_INT_MAX;
            });
            $data = MemberResource::collection($sortedMembers->values());
            return $this->sendJson($data);
        } catch (\Exception $e) {
            $this->logError($e, "Get member upcoming birthday Error");
            return $this->sendError($e->getMessage());
        }
    }

    public function userAccountDelete(Request $request)
    {
        $validator = validator($request->all(), [
            'user_id' => ['required', 'exists:users,id']
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 400);
        }

        try {
            $user = User::find($request->user_id);
            $this->logInfo("User account Request " . json_encode($user));
            Schema::disableForeignKeyConstraints();
            $user->delete();
            Schema::disableForeignKeyConstraints();
            return $this->sendJson([
                'message' => 'Your account has been successfully deleted.'
            ], 200);
        } catch (\Exception $e) {
            $this->logError($e, "User account delete Error");
            return $this->sendError($e->getMessage());
        }
    }

    public function checkMemberId(Request $request)
    {
        $data = $request->validate([
            'member_id' => ['required']
        ]);
        try {

            $user = $request->user();
            $associations = $user->associationUsers;
            $isTrue = $associations->contains(function ($associationUser) use ($data) {
                return $associationUser->user_member_id == $data['member_id'];
            });

            if ($user && $user->unique_id === $data['member_id'] || $isTrue) {
                $data = [
                    'message' => 'Verified Successfully',
                    'user_id' => $user->id ?? null,
                    'name' => $user->name ?? null,
                    'country_code' => $user->country_code ?? null,
                    'contact' => $user->contact ?? null,
                    'email' => $user->email ?? null,
                    'unique_id' => $data['member_id'],
                    'role' => $user->role ?? null,
                    'status' => $user->status ?? null,
                ];
                return $this->sendJson($data);
            }
            return $this->sendError('Invalid member id');
        } catch (\Exception $e) {
            $this->logError($e, "Verify Members ID Error");
            return $this->sendError($e->getMessage());
        }
    }
}
