<?php

namespace App\Http\Controllers\API\Event;

use App\Http\Controllers\Controller;
use App\Http\Resources\API\Event\EventDayResource;
use App\Http\Resources\API\Event\EventHallResource;
use App\Http\Resources\API\Event\EventSessionResource;
use App\Models\Event;
use App\Models\EventHall;
use App\Models\EventSession;
use Illuminate\Http\Request;

class EventSessionController extends Controller
{
    public function index(Request $request)
    {
        try {
            $event = getEventFromId($request->attributes->get('event_id'));

            if ($request->scope == 'halls') {

                $eventHalls = $event->eventHalls()->whereHas('eventSessions')->get();

                $request->merge(['display_event_session_details' => true]);
                $request->merge(['display_event_session_speaker_details' => true]);

                return EventHallResource::collection($eventHalls);
            }

            $eventDays = $event->eventDays()->whereHas('eventSessions')->get();

            $request->merge(['display_event_session_details' => true]);
            $request->merge(['display_event_session_speaker_details' => true]);

            return EventDayResource::collection($eventDays);
        } catch (\Exception $e) {
            $this->logError($e, "Get Event Session Error");
            return $this->sendError($e->getMessage());
        }
    }

    public function eventSessionDashboard(Request $request)
    {
        try {
            $request->validate(
                [
                    'type' => ['nullable', 'integer', 'in:1,2,3'],
                ]
            );
            $event = getEventFromId($request->attributes->get('event_id'));
            $eventSessions = $event->eventSessions()->whereNull('parent_id');
            if ($request->type) {
                $eventSessions->where('status', $request->type);
            }
            $eventSessions = $eventSessions->get();
            return $this->sendJson([
                'data' => EventSessionResource::collection($eventSessions),
                'registration_link' => getEnvironmentVariable('registration_link', $event->id),
            ]);
        } catch (\Exception $e) {
            $this->logError($e, "Get Upcoming Event Session Error");
            return $this->sendError($e->getMessage());
        }
    }

    public function bookmark(Request $request, EventSession $eventSession)
    {
        try {
            $eventId = $request->attributes->get('event_id');
            $user = $request->user();
            if ($eventSession->isBookmarked($user)) {
                $eventSession->removeBookmark($user);
                $message = 'Session remove from bookmark';
            } else {
                $eventSession->bookmarkEventSession($user, $eventId);
                $message = 'Session Bookmarked';
            }

            return response()->json([
                'message' => $message
            ]);
        } catch (\Exception $e) {
            $this->logError($e, "bookmark Error");
            return $this->sendError($e->getMessage());
        }
    }

    public function userBookmarkedSession(Request $request)
    {
        try {
            $eventId = $request->attributes->get('event_id');
            $user = $request->user();
            $savedBookMark = $user->bookmarkedSessions()->where('event_sessions.event_id', $eventId)->latest()->simplePaginate();
            return EventSessionResource::collection($savedBookMark);
        } catch (\Exception $e) {
            $this->logError($e, "User bookmark session error");
            return $this->sendError($e->getMessage());
        }
    }

    public function getUserSessionDetail(Request $request, Event $event)
    {
        $request->validate([
            'user_id' => ['required', 'exists:users,id'],
        ]);

        $eventSessions = EventSession::where('event_id', $event->id)
            ->where(function ($query) use ($request) {
                // Ensure each relationship filters by the `user_id` from the request
                // $query->whereHas('eventSessionUsers', function ($query) use ($request) {
                //     $query->where('user_id', $request->user_id);
                // });
                // $query->orWhereHas('eventSessionFaculties', function ($query) use ($request) {
                //     $query->where('user_id', $request->user_id);
                // });
                $query->orWhereHas('eventSessionSpeakers', function ($query) use ($request) {
                    $query->where('event_speaker_id', $request->user_id);
                });
            })
            ->get();

        dd($eventSessions);
        dd($eventSessions[0]->eventSessionSpeakers[0]->pivot);

        $eventSessions = EventSession::where('event_id', $event->id)
            ->where(function ($query) use ($request) {
                $query->orWhereHas('eventSessionUsers', function ($query) use ($request) {
                    $query->orWhere('user_id', $request->user_id);
                })->orWhereHas('eventSessionFaculties', function ($query) use ($request) {
                    $query->orWhere('user_id', $request->user_id);
                })->orWhereHas('eventSessionSpeakers', function ($query) use ($request) {
                    $query->orWhere('user_id', $request->user_id);
                });
            })->pluck('event_day_id')->toArray();

        $eventSessionsDays = $event->eventDays()->whereIn('id', $eventSessions)->get();

        $request->merge(['display_event_session_details' => true]);
        $request->merge(['display_event_session_speaker_details' => true]);

        return EventDayResource::collection($eventSessionsDays);
    }

    public function getSpeakerSessionDetail(Request $request, Event $event)
    {
        $request->validate([
            'user_id' => ['required'],
        ]);

        $eventSessions = EventSession::where('event_id', $event->id)
            ->whereHas('eventSessionSpeakers', function ($query) use ($request) {
                $query->where('event_speaker_id', $request->speaker_id);
            })
            ->get();

        $eventSessions = EventSession::where('event_id', $event->id)
            ->whereHas('eventSessionSpeakers', function ($query) use ($request) {
                $query->where('event_speaker_id', $request->speaker_id);
            })->pluck('event_day_id')->toArray();

        $eventSessionsDays = $event->eventDays()->whereIn('id', $eventSessions)->get();

        $request->merge(['display_event_session_details' => true]);
        $request->merge(['display_event_session_speaker_details' => true]);

        return EventDayResource::collection($eventSessionsDays);
    }
}
