<?php

namespace App\Helpers;

use App\Enums\MetaFileExtensionEnum;
use App\Models\Setting;
use Illuminate\Http\Client\Response;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Symfony\Component\HttpFoundation\File\File;

class MetaClient
{
    private $url = "https://graph.facebook.com";

    private $version;

    private $phoneNumberId;

    private $token;

    private $businessAccountId;

    private $appId;

    static private $_api;

    /**
     * @return MetaClient
     */
    static public function api()
    {
        if (!self::$_api) {
            self::$_api = new MetaClient();
        }
        return self::$_api;
    }

    /**
     * MetaClient constructor.
     * @throws \Exception
     */
    public function __construct()
    {
        $settings = Setting::first();
        $this->token = $settings->whatsapp_meta_token;
        $this->phoneNumberId = $settings->whatsapp_meta_phone_number_id;
        $this->version = $settings->whatsapp_meta_version;
        $this->businessAccountId = $settings->whatsapp_meta_business_account_id;
        $this->appId = $settings->whatsapp_meta_app_id;

        if (
            empty($this->token) ||
            empty($this->phoneNumberId) ||
            empty($this->version) ||
            empty($this->businessAccountId) ||
            empty($this->appId)
        ) {
            throw new \Exception("Whatsapp config not found");
        }
    }

    public function get($url)
    {
        Log::info("GET: {$url}");
        // "error": "The stream or file "/var/www/html/vdoconfrence/storage/logs/meta-2024-10-30.log 
        //could not be opened in append mode: Failed to open stream: Permission denied\nThe exception occurred while attempting to log
        // Log::channel('meta')->info("GET: {$url}");

        return Http::acceptJson()
            ->contentType("application/json")
            ->withToken($this->token)
            ->get($url);
    }

    public function post($url, $body)
    {
        Log::info("POST: {$url} - " . json_encode($body));
        // "error": "The stream or file "/var/www/html/vdoconfrence/storage/logs/meta-2024-10-30.log 
        //could not be opened in append mode: Failed to open stream: Permission denied\nThe exception occurred while attempting to log
        // Log::channel('meta')->info("POST: {$url} - " . json_encode($body));
        return Http::acceptJson()
            ->contentType("application/json")
            ->withToken($this->token)
            ->post($url, $body);
    }

    public function delete($url)
    {
        Log::info("DELETE: {$url}");
        // "error": "The stream or file "/var/www/html/vdoconfrence/storage/logs/meta-2024-10-30.log 
        //could not be opened in append mode: Failed to open stream: Permission denied\nThe exception occurred while attempting to log
        // Log::channel('meta')->info("DELETE: {$url}");
        return Http::acceptJson()
            ->contentType("application/json")
            ->withToken($this->token)
            ->delete($url);
    }

    /**
     * @param $to
     * @param $body
     * @return array|mixed
     */
    public function sendTextMessage($to, $body)
    {
        $url = "{$this->url}/{$this->version}/{$this->phoneNumberId}/messages";

        $data = [
            "messaging_product" => "whatsapp",
            "to" => $to,
            "type" => "text",
            'text' => [
                'body' => $body,
            ]
        ];

        return $this->post($url, $data);
    }

    public function createMessageTemplate($body)
    {
        $url = "{$this->url}/{$this->version}/{$this->businessAccountId}/message_templates";

        return $this->post($url, $body);
    }

    public function getMessageTemplates()
    {
        $url = "{$this->url}/{$this->version}/{$this->businessAccountId}/message_templates?fields=status&limit=10";

        return $this->get($url);
    }

    public function markRead($messageId)
    {
        $url = "{$this->url}/{$this->version}/{$this->phoneNumberId}/messages";

        $data = [
            "messaging_product" => "whatsapp",
            "status" => "read",
            "message_id" => $messageId,
        ];

        return $this->post($url, $data);
    }

    public function sendTemplateMessage($to, $templateName, $language, $components)
    {
        $url = "{$this->url}/{$this->version}/{$this->phoneNumberId}/messages";

        $data = [
            "messaging_product" => "whatsapp",
            "to" => $to,
            "type" => "template",
            'template' => [
                'name' => $templateName,
                'language' => [
                    'code' => $language,
                ],
                'components' => $components,
            ]
        ];

        return $this->post($url, $data);
    }

    public function createUploadSession($fileLength, $fileType)
    {
        $url = "{$this->url}/$this->version/{$this->appId}/uploads";

        return $this->post($url, [
            "file_length" => $fileLength,
            "file_type" => $fileType
        ]);
    }

    /**
     * @param $sessionId
     * @param File $file
     * @return Response
     */
    public function uploadMedia($sessionId, $file)
    {
        Log::info("Upload Media Session: {$sessionId}");
        // "error": "The stream or file "/var/www/html/vdoconfrence/storage/logs/meta-2024-10-30.log 
        //could not be opened in append mode: Failed to open stream: Permission denied\nThe exception occurred while attempting to log
        // Log::channel('meta')->info("Upload Media Session: {$sessionId}");
        $url = "{$this->url}/$this->version/{$sessionId}";

        return Http::withToken($this->token, "OAuth")
            ->attach('file', $file->getContent(), $file->getFilename())
            ->withHeaders([
                "file_offset" => 0
            ])
            ->post($url);
    }

    public function deleteMessageTemplate($templateId, $templateName)
    {
        $url = "$this->url/{$this->version}/{$this->businessAccountId}/message_templates?hsm_id={$templateId}&name={$templateName}";

        return $this->delete($url);
    }

    public function getMediaId($messageId)
    {
        $url = "$this->url/{$this->version}/$messageId";
        return $this->get($url);
    }

    public function getMedia($url)
    {
        return $this->get($url);
    }

    public function sendMediaMessage($to, $media, $type)
    {
        if ($type == MetaFileExtensionEnum::TYPE['image']) {
            $mediaType = "image";
        } else if ($type == MetaFileExtensionEnum::TYPE['video']) {
            $mediaType = "video";
        } else if ($type == MetaFileExtensionEnum::TYPE['document']) {
            $mediaType = "document";
        }

        $url = "{$this->url}/{$this->version}/{$this->phoneNumberId}/messages";

        $data = [
            "messaging_product" => "whatsapp",
            "to" => $to,
            "type" => $mediaType,
            $mediaType => [
                'link' => $media,
            ]
        ];
        return $this->post($url, $data);
    }
}
